#!/bin/ksh
# 
# Copyright (c) 2007, 2014, Oracle and/or its affiliates. All rights reserved.

# This script has had execution time improved based on customer feedback.

# General pkg resolver script for the 1.4+ tzupdater tool.
# Run with full path to java, after the tzupdater tool, to resynchronize the Solaris
# pkg database

# e.g. ./pkg_resolve.sh /usr/jdk/instances/jdk1.5.0_06/bin/java
#
# For 1.4+ All files in the jre/lib/zi directory are removed from pkg database,
# all files presently in that location are added.
#
# For 1.4 and later JREs, we call the FilePkgUpdater function to iterate through
# the various jre/lib/zi files and update the pkgDB with the new attributes
#
# exit 0 if no update required
# exit 1 if error encountered


function FilePkgUpdater  {

if [ x$PKG == "x" ]
then
        echo "WARNING: Update succeeded but $JAVA not directly found in package contents file."
        echo "WARNING: No package resolution performed. JDK may not have been installed via SVR4 package."
		 exit 0
fi

# Only attempt pkg updates as root:
set `/usr/bin/id`
if [ $1 != "uid=0(root)" ]; then
        echo "Not running as root: pkg refresh not attempted."
        exit 1
fi


MAINBASEDIR=`egrep "$PKG$" /var/sadm/install/contents | grep "zi " | nawk '{print $1}'`

if [ "x$MAINBASEDIR" == "x" ]
then
		 echo Cannot find base directory in /var/sadm/install/contents
		 exit 1
fi

for BASEDIR in $MAINBASEDIR
do

echo Updating package $PKG with timezone base directory $BASEDIR
echo "(update time is variable, but can take several minutes)"
#
# removef the zi files in the package:
#
# find all in & under zi directory which is also in the contents file:
# ..and removef it from PKG
# Assume everything under basedir IS in the contents file (true if pkg database 
# is consistent and this is an installed package).
# Also, nothing under this base directory should be a part of any OTHER package,
# which is a reasonable assumption.
#
# Do files and directories in one go, so as to dramatically cut execution time.
# Need to be sure to remove files in directory, before directory itself.
# Hence sort -r.

echo "Removing previous package entries..."

r=`find $BASEDIR | sort -r`
removef $PKG $r > /dev/null 
        if [ $? -ne 0 ]
        then
                echo ERROR during removef $PKG ...
                exit 1
        fi

removef -f $PKG
        if [ $? -ne 0 ]
        then
                echo ERROR during removef -f $PKG
                exit 1
        fi


# That leaves files in contents which are no longer in the filesystem.
# We have done some removef work, and finalised it with removef -f, 
# so if we now grep basedir in contents we find what is left:
#
# In most cases there will be nothing left, and removef complains if given
# an empty list, hence the if-statement.

r=`grep $BASEDIR /var/sadm/install/contents  | grep $PKG | nawk '{print $1}' | sort -r`
if [ "$r" != "" ] ; then
	removef $PKG $r > /dev/null
        if [ $? -ne 0 ]
        then
                echo ERROR during removef -f $PKG $r
                exit 1
        fi

	removef -f $PKG
        if [ $? -ne 0 ]
        then
                echo ERROR during removef -f $PKG
                exit 1
        fi

fi

# Phase 2: add current zi files to pkg directory
# find in the zi directory, 
# installf the zi-related files in the package.
#
# Again, do it all in one go, to dramatically cut execution time.

echo "Installing new package entries..."

( find $BASEDIR -type d -exec echo {} d 755 root bin \; ;
  find $BASEDIR -type f -exec echo {} v              \; ) |\
installf $PKG - > /dev/null
        if [ $? -ne 0 ]
        then
                echo ERROR during installf $PKG -
                exit 1
        fi

done  # end of loop though possible BASEDIR directories

installf -f $PKG
        if [ $? -ne 0 ]
        then
                echo ERROR during installf -f $PKG
                exit 1
        fi


echo "Completed resolving Solaris package information."
}

# main entry point:

if [[ ! -a $1 ]]
then
                 echo Non-existent java path: $1
                 exit 1
fi

JDIR=`dirname $1`
JAVA=`(cd -P $JDIR; pwd )`/java

PATH=/bin/:/usr/bin:/usr/sbin

if [ x$JAVA == "x" ]
then
		 echo "Usage: script.sh <path to java>"
		 exit 1
fi

# find package name from java path
PKG=`grep "^$JAVA " /var/sadm/install/contents | nawk '{print $10}' | sed s/\*//`

FilePkgUpdater
