Ext.namespace('Ext.ux.senior.plugins');

/**
 * Gerncia os eventos dos componentes para o que deve ser enviado para o servidor. Responsvel por conseguir as conexes, e montar os parmetros que
 * os componentes disponibilizam para o envio das mensagens
 * 
 * @author Jean.Kirchner
 */
Ext.ux.senior.plugins.EventManagerPlugin = Ext.extend(Object, {

    /**
     * Mapeamento de eventos que sero chamados dos componentes
     */
    eventMapping : null,

    /**
     * @template
     */
    init : function(owner) {
        this.owner = owner;

        this.owner.on('beforedestroy', this.onDestroy, this);
        this.owner.on('render', this.onRender, this);

        /* Apenas injeta se no existir */
        Ext.applyIf(this.owner, this.ownerOverrides);

        if (!this.owner.defaultParams) {
            this.owner.defaultParams = {};
        }

        var eventMapping = this.owner.eventMapping();

        for ( var i = 0; i < this.owner.supressEvents.length; i++) {
            var supressedEvent = this.owner.supressEvents[i];
            delete eventMapping[supressedEvent];
        }
        this.eventMapping = eventMapping;

        /*
         * Captura todos os eventos do componente e repassa para onEvent. onEvent vai ser chamado com o nome do evento por primeiro parmetro e depois
         * com todos os parmetros do evento normal do componente
         */
        Ext.util.Observable.capture(this.owner, this.onEvent, this);

    },

    /**
     * @event beforedestroy
     */
    onDestroy : function(cmp) {
        Ext.util.Observable.releaseCapture(cmp);
        if (this.map) {
            this.map.setDisabled(true);
        }
        this.owner = null;
    },

    /**
     * @event render
     */
    onRender : function() {
        /* Liga eventos de teclado */
        if (this.owner.keyMapping.length > 0) {
            var el = this.owner.getEl();

            this.map = new Ext.KeyMap(el, {
                key : this.owner.keyMapping,
                fn : this.onKeyDown,
                scope : this
            });
            
            this.map.stopEvent = this.owner.keyStopEvent != undefined ? this.owner.keyStopEvent : true;
            
        }
    },

    /**
     * Qualquer evento que ocorrer no componente vai passar por aqui
     * 
     * @private
     */
    onEvent : function(eventName) {
        var objMethod = this.eventMapping[eventName];
        /* Se no tem o mtodo, o componente no trata ento simplesmente deixa o evento fluir (return true) */
        if (!objMethod) {
            return true;
        }

        /*
         * transforma o arguments em um array, name no  necessrio
         */
        var args = Array.prototype.slice.call(arguments, 1);

        var messageListener = null;
        for (var i = 0; i < args.length; i++) {
            var obj = args[i];

            if (Ext.isObject(obj) && obj instanceof ConnectionMessageListener) {
                messageListener = obj;
                args.splice(i, 1);
                break;
            }
        }

        var cmpData = objMethod.apply(this.owner, args); /* Manda o componente preparar os dados */

        return this.sendData(cmpData, eventName, args, messageListener);

    },

    sendData : function(cmpData, eventName, args, messageListener) {
        var block = SProfilerManager.start(eventName);

        var args = Array.prototype.slice.call(arguments, 2);

        /* S continua se o componente tratou o evento, ou seja preparou algum dado para envio */
        if (!cmpData) {
            SProfilerManager.stop(block);
            return true;
        }

        /* Verificar a existncia das propriedades necessrias */
        if (!this.owner.defaultParams) {
            SProfilerManager.stop(block);
            throw String.format("defaultParams does not exist, did you remove it by accident?. Dump: componenteId: [{0}], xtype: [{1}], eventName: [{2}]", this.owner.getId(), this.owner.xtype,
                    eventName);
        }

        if (!this.owner.getEventTarget) {
            SProfilerManager.stop(block);
            throw String.format("getEventTarget does not exist, did you remove it by accident?. Dump: componenteId: [{0}], xtype: [{1}], eventName: [{2}]", this.owner.getId(), this.owner.xtype,
                    eventName);
        }

        this.assertApply(cmpData, this.owner.defaultParams, eventName);

        /* Monta os dados comuns a qualquer request */
        args.unshift(this.owner); /* joga o componente como primeiro parmetro */
        var params = EventDataBuilder.buildData.apply(EventDataBuilder, args);
        this.assertApply(cmpData, params, eventName);

        /* Parmetros do componente, apenas o type vai na raiz */
        var reqObj = {
            id : this.owner.getEventTarget(),
            type : cmpData.type,
            eventParams : cmpData,
            messageListener : messageListener
        };

        var connectionQueue = cmpData['connection'] || ConnectionManager.DEFAULT_URL;
        delete cmpData.type;
        delete cmpData.connection;

        /* Manda o request para o servidor */
        var conn = ConnectionManager.connectionFor(this.owner);
        reqObj.target = conn.context;
        
        conn.connection[connectionQueue].enqueueRequest(reqObj);

        /* se retornar false vai cancelar o evento */
        SProfilerManager.stop(block);
        return !cmpData.cancel;
    },

    /**
     * Aplica propriedades de params para o cmpData Se alguma propriedade j existir, dispara uma assertiva
     * 
     * Isso ajuda a no ocorrer conflitos quando componentes querem enviar propriedades que j so utilizadas mais genricamente para todoso os
     * eventos
     * 
     * @private
     */
    assertApply : function(cmpData, params, xtype, eventName) {
        for ( var prop in params) {
            if (cmpData[prop]) {
                throw String.format("This property [{0}] in params, is already in use for component [{1}], at the event [{2}]", prop, this.owner.xtype, eventName);
            }
            cmpData[prop] = params[prop];
        }
    },

    /**
     * Teclas mapeadas pelo componente cairo neste mtodo que ser repassado para o componente
     * 
     * @event KeyMap
     */
    onKeyDown : function(key, evObj) {
        var cmpData = this.owner.onKeyEvent.apply(this.owner, [ key, evObj ]);

        /* Se no tem o mtodo, o componente no trata ento simplesmente deixa o evento fluir (return true) */
        if (cmpData) {
            this.sendData(cmpData, "keyPressed");
        }

        /**
         * Existe um problema no IE, que no  possvel parar um evento (o evento ainda vai para o IE) no qual o browser tem atalhos para a tecla em
         * questo. Nestes casos basta adicionar um case para que seja feito este processamento
         */
        if (Ext.isIE) {
            switch (key) {
            /* Neste caso, a situao  com a tecla F3 alguns browsers comeam uma pesquisa */
            case Ext.EventObject.F3:
                /* Neste caso, a situao  com a tecla F4 que coloca foco na barra de endereos */
            case Ext.EventObject.F4:
                /* altera-se a tecla para o browser no processar */
                window.event.keyCode = 0;
                break;
            }
        }

    },

    /**
     * @template
     */
    ownerOverrides : {
        /**
         * Todo componente que desejar fazer algum tratamento especfico para algum evento, deve especificar esta propriedade. O mtodo que trata o
         * evento deve retornar um objeto apenas com as propriedades desejadas para serem tratadas pelo seu controler. Se for necessrio cancelar o
         * evento, basta retornar junto com o objeto de requisio a propriedade "cancel"
         * 
         * Exemplo de mapeamento de eventos: <br>
         * 
         * eventMapping: function(){ return { click: this.onClick, rowselect: this.onRowSelect }; <br> }
         * 
         * Nesse exemplo vamos tratar os eventos click e rowselect, que seria equivalente no ext : <br>
         * this.on("click", this.onClick)
         * 
         */
        eventMapping : function() {
            return {};
        },

        /**
         * Array vazio para o caso do componente no ter nenhuma tecla mapeada
         */
        keyMapping : [],

        /**
         * Quando algum componente precisar de algum evento de teclado deve fazer override deste mtodo, alm de informar a propriedade keyMapping,
         * para todas as teclas desejadas para serem interceptadas.
         * 
         * Ex: keyMapping: "abc" //onKeyEvent vai ser chamada quando a, b ou c forem precionados.
         * 
         * Ex2: keyMapping: "\t\n" //onKeyEvent vai ser chamada quando tab ou enter forem precionados.
         * 
         * Ex3: keyMapping: [Ext.EventObject.F4, Ext.EventObject.ENTER] //onKeyEvent vai ser chamada quando F4 ou ENTER for precionado
         * 
         * 
         * @param keycode
         *            (Number) a tecla que o evento foi disparado
         * @param event
         *            Event evento disparado, quando precisar de informaes adicionais (ctrl, alt, shift, tecla, target, etc)
         * 
         */
        onKeyEvent : Ext.emptyFn,

        /**
         * Propriedades default que sero enviadas em todos os requests. Se algum componente precisar pode-se definir estas propriedades da seguinte
         * forma:
         * 
         * defaulParams: { some_prop : value }
         * 
         * Alm disto, outros componentes podem modificar o defaultParams. Por exemplo a grid, quando tiver editores, sempre vai precisar enviar algum
         * parmetro para tratar isto, ento o editor adiciona propriedade no defaultParams.
         * 
         * O uso "deve ser da seguinte forma" quando utilizado externamente:
         * 
         * this.owner.defaultParams.prop = value;
         * 
         * Onde owner  apenas um exemplo, pode ser qualquer componente. Enfatizando "No fazer override do defaultParams, ou remov-lo ou mesmo
         * exclu-lo, isto poder acarretar em problemas".
         * 
         */
        defaultParams : undefined,

        /**
         * Obtm o id do alvo do evento no servidor. Normalmente o id do prprio componente no cliente j corresponde ao id do controller do
         * componente no servidor, mas podem haver casos em que um componente deseja especificar quem ir tratar seus eventos no servidor. Para isto,
         * o componente pode definir a sua prpria funo "getEventTarget". Por exemplo:<br>
         * <br>
         * 
         * getEventTarget: function() { return 'idTratadorEvento'; }
         */
        getEventTarget : function() {
            return this.getId();
        },

        /**
         * Lista de eventos que sero "cancelados" do "eventMapping".<br>
         * Quando algum dos eventos que o componente trata especificamente, for executado por outra rotina<br>
         * Ex.: A rotina de "change" do Field  executada pelo GridEditor quando o Field estiver dentro de uma<br>
         * Grid, logo, o evento de "change" no deve ser executado quando o Field estiver em uma Grid. Para isso<br>
         * deve ser configurado na renderizao do field a lista de "supressEvents" para que ele possua o mtodo "change".
         */
        supressEvents : []
    }
});

Ext.preg('eventmanager', Ext.ux.senior.plugins.EventManagerPlugin);