/**
 * UploadRequestMessage representa o formato dos requests
 * 
 * Formato do request: <?xml version=\"1.0\" encoding=\"ISO-8859-1\"?> <request perspectiveId="afsdfsdf"> <message target="fragmento que vai receber a mensagem">
 * <id>"id_do_componente_que_gerou_o_evento"</id> <type>"tipo_de _evento"</type> <properties> <!-- no valor da proprieade vamos usar o CDATA do xml e escape(valor), para tratamentos de caracteres
 * especiais --> <prop name="nome prop"/>"valor da propriedade"</prop> </properties> </message> </request>
 */
function UploadRequestMessage(perspectiveId) {
    this.perspectiveId = perspectiveId;
    this.request = new Array();
}

UploadRequestMessage.prototype = {

    constructor : UploadRequestMessage,

    /**
     * Apenas faz verificaes e lana um erro se necessrio
     */
    check : function(msg) {
        /* comea de 1, pois o msg no  necessrio verificar */
        for ( var i = 1; i < arguments.length; i++) {
            /* tem que considerar "" portanto no utiliza-se !arguments[i]*/
            if (arguments[i] == null || arguments[i] == undefined) {
                /* Limpa quando da erro */
                this.clear();
                throw new Error(msg);
            }
        }
    },

    /**
     * Limpa o anterior para comear um novo, reaproveitando o objeto
     */
    newRequest : function() {
        this.clear();
    },

    /**
     * Cria um novo message para o request
     */
    newMessage : function(messageData) {
        this.check("Mensagem deve ter fileField", messageData.eventParams.fileField);

        this.request.push(messageData);
    },

    /**
     * Limpa os dados dessa mensagem, deixando-a pronta para ser reutilizada como se fosse uma nova mensagem.
     * 
     * @private
     */
    clear : function() {
        this.request = new Array();
    },

    getSize : function() {
        return this.request.length;
    }

};

/**
 * Gerenciador da conexo com o servidor. Ao mandar enviar uma mensagem, ela ser guardada para ser enviada assim que o servidor estiver disponvel. Se mais mensagens foram requisitadas, elas sero
 * enviadas em um nico request.
 * 
 * @param url
 *            recurso solicitado na url sem as /, ex.: start/, message/, destroy/?perspective....
 */
function UploadConnection(url, perspectiveId) {
    this.reqMsg = new UploadRequestMessage(perspectiveId);
    this.reqMsg.newRequest();

    this.url = url;
    this.connectionIdle = true;
    /*
    * Necessrio duas listas, de forma que antes de vir a resposta de um
    * request, mais requests podem ter sido enfileirados Sendo assim necessrio
    * salvar os estados dos listeners no qual foram chamados os beforeRequest
    * para chamar os afterRequests Sem chamar o after request dos listeners
    * para os requests que ainda no comearam
    */
    this.listeners = new Array();
    this.afterListeners = null;
    this.serverIdle = 0;

};

UploadConnection.prototype = {

    constructor : UploadConnection,

    release : function() {
        this.reqMsg = null;
        this.url = null;
    },

    /**
     * Para utilizar este metodo, o objeto obsComp deve ter no mnimo um destes mtodos: beforeRequest, afterRequest
     * 
     * Antes de ser feito o request para o servidor ser chamado o mtodo beforeRequest se houver. E aps o request 
     * ser chamado afterRequest. Aps ler a resposta do servidor todos os listerners sero removidos
     * 
     * @param obsComp
     *            o listener para o request corrente
     */
    observe : function(obsComp) {
        this.listeners.push(obsComp);
    },

    sendRequest : function() {
        if (this.reqMsg.getSize() == 0) {
            /* Nada a processar */
            return;
        }
        if (!this.connectionIdle || !this.messageIsReady()) {
            this.checkRequests();
            return;
        }

        this.connectionIdle = false;

        // Guardamos a mensagem que est sendo processada porque ela possui
        // os listeners que sero disparados aps a execuo da mensagem.
        this.processingMsg = this.reqMsg.messageListeners && this.reqMsg.messageListeners.slice();
        
        this.beforeRequest();

        var uploadHandler = new qq.FileUploader({
            action : 'upload',
            debug : true,
            multiple : false,
            encoding : 'multipart',
            inputName : 'upload',
            uploadConnection : this,

            // Mtodo onComplete  chamado sempre ao final do upload, com ou sem sucesso do upload.
            // Para os browsers que fazem a transmisso via XMLHttpRequest (Chrome, Firefox, IE10+):  disparado o evento onError no caso de falhas.
            // Para os browsers que fazem a transmisso via iframe (IE9-):  disparado somente o mtodo onComplete.
            // Importante: a resposta do servidor tem que ser SC_OK (200), seno o IE9 lana o erro "access denied" ao tentar ler o response do iframe.
            // -------------------------------------------------------------------------------------
            // tl;dr: resposta do servidor tem que ser sempre 200 (ok), sobreescrever somente o mtodo onComplete para adicionar comportamentos.
            onComplete : function(id, fileName, responseJSON) {
                // se no possui mais nenhuma conexo com o servidor
                // ento a conexo ficou idle novamente e notifica os listeners.
                if (--this.uploadConnection.serverIdle == 0) {
                    this.uploadConnection.connectionIdle = true;
                    this.uploadConnection.afterRequest();
                }

                var request = paramMap[id];
                var listener = request.messageListener;
                if (listener && listener.onComplete) {
                    listener.onComplete(!responseJSON.errorMessage, fileName, responseJSON);
                }
            }
        });

        // mapa utilizado para identificarmos a requisio dentro da function onComplete
        var paramMap = {};

        var requests = this.reqMsg.request;
        for ( var i = 0; i < requests.length; i++) {
            this.serverIdle++;

            var file = requests[i].eventParams.fileField;
            var id = uploadHandler.uploadFile(file);
            paramMap[id] = requests[i];
        }

        /* Aps enviar as mensagens pendentes comea uma nova */
        this.reqMsg.newRequest();
    },

    /**
     * Retorna <code>true</code> caso todos os listeners da mensagem permitam a sua execuo.
     * 
     * @returns {Boolean}
     */
    messageIsReady : function() {
        var messageListeners = this.reqMsg.messageListeners;
        if (messageListeners) {
            for ( var i = 0; i < messageListeners.length; i++) {
                var messageListener = messageListeners[i];
                if (messageListener.canSendRequest && messageListener.canSendRequest != undefined) {
                    if (!messageListener.canSendRequest()) {
                        return false;
                    }
                }
            }
        }
        return true;
    },

    notifyMessageListener : function(success) {
        var messageListeners = this.processingMsg;
        if (messageListeners) {
            for ( var i = 0; i < messageListeners.length; i++) {
                var messageListener = messageListeners[i];
                if (success) {
                    if (messageListener.onSuccess && messageListener.onSuccess != undefined) {
                        messageListener.onSuccess();
                    }
                } else {
                    if (messageListener.onFailure && messageListener.onFailure != undefined) {
                        messageListener.onFailure();
                    }
                }
            }
        }
        this.processingMsg = null;
    },

    /**
     * Chamado antes de fazer o request Chama o evento beforeRequest dos listeners se existir
     */
    beforeRequest : function() {
        for ( var i = 0; i < this.listeners.length; i++) {
            if (this.listeners[i].beforeRequest && this.listeners[i].beforeRequest != undefined) {
                this.listeners[i].beforeRequest();
            }
        }
        /* Faz o backup dos listeners atuais para serem chamados posteriormente */
        /* quando vier a resposta do request */
        this.afterListeners = this.listeners;
        /* Cria um novo array para os novos listeners de novos requests */
        this.listeners = new Array();
    },

    /**
     * Chamado aps ser feito o request Chama o evento afterRequest dos listeners se existir TODO: verificar necessidade de passar um parmetro sucess (se o request foi efetuado com sucesso ou no)
     */
    afterRequest : function() {
        if (!this.afterListeners) {
            return;
        }
        for ( var i = 0; i < this.afterListeners.length; i++) {
            if (this.afterListeners[i].afterRequest && this.afterListeners[i].afterRequest != undefined) {
                this.afterListeners[i].afterRequest();
            }
        }
        this.afterListeners.splice(0, this.afterListeners.length - 1);
        /* Mata o afterListeners sendo que receber listeners posteriormente */
        this.afterListeners = null;
    },

    /**
     * @private
     * Trata erro de conexo, tratamentos: 
     *  - Quando vier a tela de login como request, recarrega a pgina
     *  - Quando vier a tela de erro crtico como request, executa a funo que veio na tela  
     */
    handleError : function(response, exception) {
        throw exception;
    },

    /**
     * @param reqObj
     *            json no seguinte formato: 
     *            { 
     *              target: '', 
     *              id: '', 
     *              type: '', 
     *              eventParams: { estrutura do componente, com todas as propriedades na raiz, e podendo utilizar arrays }
     *            }
     */
    enqueueRequest : function(reqObj) {
        this.addMessage(reqObj);

        var listener = {};
        listener.afterRequest = function() {
            ConnectionManager.getConnection().addUserActionControlMessage();
        };
        this.observe(listener);

        if (this.connectionIdle) {
            /* indica que est processando uma ao do usurio */
            ConnectionManager.setUserActionProcessing();
            this.sendRequest();
        } else {
            this.checkRequests();
        }
    },

    enqueueProtocolRequest : function(reqObj) {
        this.addMessage(reqObj);

        if (this.connectionIdle) {
            this.sendRequest();
        } else {
            this.checkRequests();
        }
    },

    addMessage : function(messageData) {
        this.reqMsg.newMessage(messageData);
    },

    isServerIdle : function() {
        return this.serverIdle == 0;
    },

    isConnectionIdle : function() {
        return this.connectionIdle;
    },

    /**
     * Verifica se alguma requisio precisa ser enviada, chamando o sendRequest
     */
    checkRequests : function() {
        this.sendRequest.defer(1, this);
    },

    logoff : function() {
        this.reqMsg.newRequest();
    }
};
