/**
 * Responsável por gerenciar as conexões para os contextos e para os componentes
 * no browser.
 * 
 * Os contextos serão as entidades que poderão receber mensagens no servidor, e
 * que tem um componente que o representa no client.
 * 
 * Os componentes conseguirão a conexão pela hierarquia, sendo que os
 * componentes estarão dentro de um contexto pela hierarquia, no entanto há uma
 * exceção que existem componentes que não tem um "pai", mas mesmo assim
 * trabalham em conjunto com algum outro componentes, portanto ainda pertencem a
 * um contexto. Então estes componentes terão uma propriedade chamada
 * componentOwner, no qual terá outro componente no qual está auxiliando, e
 * assim será possível conseguir uma conexão para um componente orfão.
 * 
 * @author jean.kirchner
 */
ConnectionManager = {
        
    /** @public - URL padrão para os requests */
    DEFAULT_URL : "message",
    /** @public - URL padrão para os requests relacionados ao upload */
    UPLOAD_URL : "upload",
    /** @private - Os contextos com as conexões correspondentes */
    ctxConnections : [],
    /** @private - Os componentes com as conexões correspondentes */
    cmpConnections : [], 

    /** @private - conexão */
    conn : null,

    /** @private - contexto, o id da perspectiva */
    context : null,

    /** @private - se tem ação de usuário sendo processada */
    userActionProcessing : null,
    
    /** @private - número da ultima mensagem enviada */
    lastMessageNumber : 0,

    /** @private - número da mensagem que esta sendo executada */
    currentMessageNumber : 0,

    /**
     * Registra os contextos existentes no client, necessário para um mapeamento
     * de conexões
     * 
     * @param {Object}
     *            context - Id do contexto
     * @param {Object}
     *            cmpId - Id do componente que representa tal contexto
     */
    registerContext : function(context, cmpId) {
        if (!this.conn) {
            this.conn = {};
            this.conn[this.DEFAULT_URL] = new DefaultConnection(this.DEFAULT_URL, context);
            this.conn[this.UPLOAD_URL]  = new UploadConnection(this.UPLOAD_URL, context);
        }

        if (this.context && this.context != context) {
            throw new Error("Context diferente do configurado, atual: " + this.context + ", novo: " + context);
        }
        this.context = context;

        if (cmpId) {
            if(this.cmpConnections[cmpId]) {
                this.cmpConnections[cmpId].connection = this.conn;
                this.cmpConnections[cmpId].context = this.context;
            } else {
                this.cmpConnections[cmpId] = new ConnectionRef(this.conn, this.context);
            }
        }
    },

    /**
     * Retorna a conexão. Pode não estar criada ainda.
     */
    getConnection : function(connection) {
        var conn = this.conn[connection];
        return conn || this.conn[this.DEFAULT_URL];
    },
    
    /**
     * Retorna o ConnectionManager.
     * <p>
     * Método criada para conseguir definir um escopo através da API do Formcenter.
     * Ver implementação das classes: RenderCurrentMessageAction, RenderUserActionDone e RenderServerIdle.
     */
    getConnectionManager : function() {
        return this;
    },
    
    /**
     * Retorna o contexto. Pode estar nulo.
     */
    getContext : function() {
        return this.context;
    },

    /**
     * Registra uma conexão já existente de fromid para toId
     * 
     * @param {Object}
     *            fromId - De quem pegar a conexão
     * @param {Object}
     *            toId - Para quem registrar
     */
    registerConnectionFrom : function(toCtx, cmpId) {
        if (!toCtx) {
            throw new Error("Deve existir um fromId e um toId para a conex�o ser registrada");
        }
        if (this.cmpConnections[cmpId]) {
            this.cmpConnections[cmpId].connection = this.conn;
            this.cmpConnections[cmpId].context = toCtx;
        } else {
            this.cmpConnections[cmpId] = new ConnectionRef(this.conn, toCtx);
        }
    },

    findAndRegisterConnectionFrom : function(toCtx, cmpId) {
        var connection = this.connectionFor(Ext.getCmp(toCtx));
        this.cmpConnections[cmpId] = connection;
        
        if (Senior.isRunningTest()) {
            Selenium.put(toCtx, cmpId);
        }
    },
    
    /**
     * Consegue uma conexão para um componente
     * 
     * @param {Component}
     *            extCmp - O componente para conseguir o contexto
     * @return um conector para o componente
     */
    connectionFor : function(cmp) {
        if (!cmp) {
            throw new Error("cmp can't be null");
        }

        /* Se o componente ainda não tiver uma conexão */
        if (!cmp.conn) {
            /* Acha o primeiro container direto com conexão */
            var extCmp = cmp;
            while (extCmp && !this.cmpConnections[extCmp.getId()]) {
                extCmp = extCmp.ownerCt;
            }
            if (!extCmp) {
                /*
                 * Em alguns casos um componente não tem nenhum pai, mas pode
                 * pertencer a algum outro componente como utilitário
                 */
                /*
                 * E se ainda assim, não tendo nenhum pai e nenhum dono, dai
                 * dispara erro
                 */
                if (!cmp.componentOwner) {
                    return this.handleConnectionNotFound(cmp);
                }

                /* Pega a conexão do dono deste componente */
                var conn = this.connectionFor(Ext.getCmp(cmp.componentOwner));
                cmp.conn = conn;
            } else {
                /* E usa a mesma conexão deste contexto */
                cmp.conn = this.cmpConnections[extCmp.getId()];
            }
        }
        /* Retorna a conexão com contexto; */
        return cmp.conn;
    }, 
    
    handleConnectionNotFound : function(cmp) {
        throw new Error("Connection not found for " + cmp);
    },
    
    isServerIdle : function() {
        if (!this.conn) return true;
        for (var property in this.conn) {
            if (!this.conn[property].isServerIdle()) 
                return false;
        }
        return true;
    },

    isUserActionProcessing : function() {
        return this.userActionProcessing;
    },

    setUserActionDone : function(requestNumber) {
        /*se a resposta foi do ultimo pacote enviado, então o controle está com o client*/
        if (this.lastMessageNumber == requestNumber) {
            this.userActionProcessing = false;
        }
    },

    setUserActionProcessing : function() {
        this.userActionProcessing = true;
    },

    incrementAndGetLastMessageNumber : function() {
        return ++this.lastMessageNumber;
    },

    /**
     * Seta qual o atual request que esta sendo executando
     * @param currentMessageNumber número do request
     */
    setCurrentMessageNumber : function(currentMessageNumber) {
        this.currentMessageNumber = currentMessageNumber;
    },

    /**
     * Encontra qua o número do request que esta sendo executado
     * @returns {Number}
     */
    getCurrentMessageNumber : function() {
        return this.currentMessageNumber;
    }

};

function ConnectionRef(connection, context) {
    this.connection = connection;
    this.context = context;
};

ConnectionRef.prototype = {
    constructor : ConnectionRef
};