Ext.namespace('Ext.ux.senior.timer');

/**
 * Responsvel pelas requisies de execuo da ao do Timer no servidor.
 * Pode ser iniciado pela funo <code>start</code>. Depois de iniciado,
 * fica enviando requisies para o servidor a cada <code>interval</code>. O request  tratado pelo <code>Timer</code>
 * do servidor, que ir executar a ao configurada para ele (apenas se j no estiver executando).<br>
 * Para interromper o Timer, a funo <code>stop</code> deve ser chamada.<br>
 * Se o Timer no for mais utilizado, deve ser destrudo atravs da funo <code>destroy</code>.<br>
 * <br>
 * Todo instncia de Timer  registrada automaticamente no <code>TimerManager</code>. Esta instncia pode ser
 * obtida posteriormente atravs do mtodo <code>get</code> do <code>TimerManager</code>.
 */
Ext.ux.senior.timer.Timer = function(id, interval) {
	this.id = id;
	
	//Cria a task que envoa o request para o servidor.
	this.task = {
		interval: interval,
		scope: this
	};
	
	//Registra no TimerManager
	Ext.ux.senior.timer.TimerManager.register(this);
};

/**
 * Funo que ir rodar a cada intervalo do timer, enviando requisies ao servidor
 * para que a operao do Timer de l seja executada.
 */
Ext.ux.senior.timer.Timer.run = function() {
    var reqobj = {
        target : this.getId(),
        type : "executeTask",
        id : "null"
    };
    ConnectionManager.getConnection().enqueueProtocolRequest(reqobj);
},

Ext.ux.senior.timer.Timer.firstRun = function() {
	/*
	 * Na primeira vez no faz nada porque o ext executa a task logo que inicia ela.
	 * Desta forma, ele vai esperar o intervalo para executar a task. 
	 */
	this.task.run = Ext.ux.senior.timer.Timer.run;
},

Ext.ux.senior.timer.Timer.prototype = {
		
	started : false,
	
	/**
	 * Inicia o <code>Timer</code> para que ele fique enviando requests ao servidor a cada intervalo de tempo, determinado pela
	 * propriedade <code>interval</code>. O request  tratado pelo <code>Timer</code> do servidor, que ir executar
	 * a ao configurada para ele (apenas se j no estiver executando).<br>
	 * 
	 * @param {Boolean} runOnStart Define se  para executar a operao do Timer assim que ele  iniciado. O padro  <code>true</code>.
	 * @public 
	 */
	start : function(runOnStart) {
		if (this.started) {
			throw String.format('The timer "{0}" is already started', this.id);
		}
		
		if (this.getInterval() <= 0) {
			throw String.format("Invalid interval: {0}", interval);
		}
		
		//O TaskMgr j executa o "run" logo que inicia, por isso para fazer com que a operao s seja executada
		//depois do intervalo,  necessrio ignorar a primeira execuo.
		if (runOnStart !== false) {
			this.task.run = Ext.ux.senior.timer.Timer.run;
		} else {
			this.task.run = Ext.ux.senior.timer.Timer.firstRun;
		}
		Ext.TaskMgr.start(this.task);
		
		this.started = true;
	},
	
	/**
	 * Interrompe o timer.
	 * 
	 * @public
	 */
	stop : function() {
		if (!this.started) {
			throw String.format('The timer "{0}" is not started', this.id);
		}
		
		Ext.TaskMgr.stop(this.task);
		
		this.started = false;
	},
	
	/**
	 * Define o intervalo de tempo em que os requests so feitos para o servidor.
	 * 
	 * @param interval intervalo de tempo
	 * @public
	 */
	setInterval : function(interval) {
		if (interval <= 0) {
			throw String.format("Invalid interval: {0}", interval);
		}
		
		this.task.interval = interval; 
	},
	
	/**
	 * Interrompe o timer se ele estiver iniciado e remove seu registro do <code>TimerManager</code>.
	 * 
	 * @public
	 */
	destroy : function() {
		//Se estiver iniciado, para o timer.
		if (this.started) {
			this.stop();
		}
		
		//Remove o registro do TimerManager
		Ext.ux.senior.timer.TimerManager.unregister(this);
	},
	
	/**
	 * @returns O id do <code>Timer</code>.
	 * @public
	 */
	getId : function() {
		return this.id;
	},
	
	/**
	 * @returns O interval do <code>Timer</code>.
	 * @public
	 */
	getInterval : function() {
		return this.task.interval;
	}
};
