/**
 * @class Ext.layout.FloatLayout
 * @extends Ext.layout.BoxLayout
 * <p>A layout that arranges items in a left to right flow similar to using HTML's <code>float: left</code> style.</p>
 * <p>When a Component's width (including its padding) won't fit within the available width of the Container (within its padding)
 * then the Component is wrapped to the beginning of a new line, clearing the tallest item on the current line.</p>
 * <p>Items may be vertically aligned with the row they happen to fall within. A line's height is the height of its tallest child
 * item (including its margins), and vertical alignment take place within this space. Vertical alignment is specified for the layout using the
 * {@link #verticalAlign} config, and may be overriden for a single item by configuring an item with a verticalAlign value.</p>
 * <p>Row content may be aligned within the available width of the Container using the {@link #horizontalAlign} config.</p>
 * <p>Child items may be cleared left or right by configuring a <b>clear</b> value in the item.</p>
 * <p>Row and column spacing may be specified using the {@link #verticalSpacing} and {@link #horizontalSpacing} options.
 */
Ext.layout.FloatLayout = Ext.extend(Ext.layout.BoxLayout, {
    /**
     * @cfg {Boolean} animate
     * <p><code>true</code> to animate child items into their calculated positions.</p>
     * <p>Defaults to <code>false</code>.</p>
     */

    /**
     * @cfg {String} verticalAlign
     * <p>Vertical alignment of child items within each row. Defaults to 'middle'. Values may be</p>
     * <code>'top'</code>, <code>'middle'</code> or <code>'bottom'</code>
     * <p>May be overriden by an individual child item.</p>
     */
    verticalAlign : 'top',

    /**
     * @cfg {String} horizontalAlign
     * <p>Horizontal alignment of child items within each row. Defaults to 'left'. Values may be</p>
     * <code>'left'</code>, <code>'center'</code>, <code>'right'</code> or <code>'justify'</code>
     * <p>May be overriden by an individual child item.</p>
     */
    horizontalAlign : 'left',

    /**
     * @cfg {Number} horizontalSpacing
     * <p>Horizontal space to leave between items.</p>
     */
    horizontalSpacing : 0,

    /**
     * @cfg {Number} verticalSpacing
     * <p>Vertical space to leave between rows.</p>
     */
    verticalSpacing : 0,

    innerCls : null,

    targetCls : 'ux-float-layout-ct',

	overflowHandler : 'Float',

    renderItem : function(c) {
        Ext.layout.FloatLayout.superclass.renderItem.apply(this, arguments);
        
        if (c.xtype == 'seniorimage') {
            
            if(!c.hasListener('load'))
                c.image.on('load', function(){
                    if(this.rendered && this.getVisibilityEl().isVisible(true)){
                        this.doLayout();
                    } else {
                        // est desenhando uma imagem num container no visvel - algum para cima deve estar 'colapsado'
                        var container = this.findParentBy(function(container, item){
                            if(container.collapsed) 
                                return true;
                            return false;
                        });
                        
                        // flag que vai evitar adicionar 2x o evento de expand no mesmo container caso ele tenha mais de 1 imagem.
                        if(container.hasExpandOnLoadImageEvent !== true){
                            container.on("expand", function(){
                                this.doLayout();
                                this.hasExpandOnLoadImageEvent = false;
                            }, container, {
                                single : true 
                            }); 
                            container.hasExpandOnLoadImageEvent = true;                
                        }
                    }
                }, this.container);
        }
    },

    alignLeftToRight : function(b) {
        //a largura do container
        var width = this.container.getLayoutTarget().getWidth();
        //guarda o valor em que deve estar o x do prximo componente
        var lastX = 0;
        //indica o valor do ltimo y adicionado(diferente do lastX, s serve pra confirmar se a linha continua a mesma)
        var lastY = -1;
        this.container.getLayoutTarget().applyStyles('position:relative');
        for ( var i = 0, l = b.length; i < l; i++) {
            if (this.animate) {
                Ext.lib.Anim.motion(b[i].component.getPositionEl().dom, {
                    left : {
                        to : b[i].left
                    },
                    top : {
                        to : b[i].top
                    }
                }).animate();
            } else {
                b[i].component.setPosition(b[i].left, b[i].top);
            }
        }

    },

    alignRightToLeft : function(b) {
        /*
         * Quando o alinhamento  da direita para esquerda no  possvel adicionar 
         * o "float: right" pois ele inverte a sequencia dos componentes.
         * Sendo assim os componentes so colocados um a um alinhando-os  direita,
         * garantindo que a sequencia de declarao seja mantida. 
         */
        this.container.getLayoutTarget().applyStyles('position:relative');

        //a largura do container
        var width = this.container.getLayoutTarget().getWidth();
        //guarda o valor em que deve estar o x do prximo componente
        var lastX = width;
        //indica o valor do ltimo y adicionado(diferente do lastX, s serve pra confirmar se a linha continua a mesma)
        var lastY = 0;
        // o maior componente (altura) da linha
        var mostHeight = 0;
        // um array que guarda os componentes por linha
        var line = [];
        for ( var i = 0, l = b.length; i < l; i++) {
            lastX -= b[i].width;
            if (mostHeight < b[i].height) {
                mostHeight = b[i].height;
            }
            if (lastX - b[i].width <= -1) {
                // quando acaba o espao na linha, configura o X/Y dos componentes
                // a ordem de declarao  mantida,  somente feito alinhamento  direita
                for ( var j = 0; j < line.length; j++) {
                    line[j].left = lastX;
                    line[j].top = lastY;
                    lastX += line[j].width;
                }
                // criao para a segunda linha
                line = [];
                lastY += mostHeight;
                lastX = width - b[i].width;
                mostHeight = 0;
            }

            // adiciona o componente corrente na linha
            line[line.length] = b[i];

            if (this.animate) {
                Ext.lib.Anim.motion(b[i].component.getPositionEl().dom, {
                    left : {
                        to : b[i].left
                    },
                    top : {
                        to : b[i].top
                    }
                }).animate();
            } else {
                b[i].component.setPosition( [ b[i].left, b[i].top ]);
            }
        }
    },

    updateChildBoxes : function(b) {
        var isRightAlign = this.horizontalAlign == 'right';
        if (!isRightAlign) {
            this.alignLeftToRight(b);
        } else {
            this.alignRightToLeft(b);
        }
    },

    getVisibleItems : function(ct) {
        var items = ct.items;
        var visibleItems = [];
        for ( var i = 0; i < items.length; i++) {
            var item = items.get(i);
            if (!item.hidden) {
                visibleItems.push(item);
            }
        }
        return visibleItems;
    },

    /**
     * Calcula a altura, largura, topo e esquerda dos componentes visveis deste container
     * 
     * @returns Array contendo o clculo de altura, largura, topo e esquerda de cada componente visel 
     */
    calculateChildBoxes : function() {
        var container = this.container;
        var layoutTarget = container.getLayoutTarget();
        var paddingLeft = this.padding.left + layoutTarget.getPadding('l');
        var paddingTop = this.padding.top + layoutTarget.getPadding('t');
        var styleSize = layoutTarget.getStyleSize();
        var rightLimit = (styleSize.width -= (this.padding.right + layoutTarget.getPadding('r')));
        var bottomLimit = styleSize.height - (this.padding.bottom + layoutTarget.getPadding('b'));
        var rowStart = 0;
        var rowHeight = 0;
        var rowWidth = 0;
        var x = paddingLeft;
        var y = paddingTop;
        var items = this.getVisibleItems(container);
        var component;
        var size;
        var right;
        var boxes = [];

        //      We never want to see horizontal scroll bars if possible
        layoutTarget.setStyle('overflow-x', 'hidden');

        //      Content Width.
        styleSize.width -= paddingLeft;

        for ( var i = 0, length = items.length; i < length; i++) {
            component = items[i];
            if (Ext.isFunction(component.doLayout)) {
                component.doLayout();
            }

            size = component.getSize();
            size.width += component.getPositionEl().getMargins('lr');
            size.height += component.getPositionEl().getMargins('tb');
            right = x + size.width;

            //          This item won't fit on the row.
            if ((right > rightLimit) || (component.clear == 'left') || ((i > 0) && items[i - 1].clear == 'right')) {
                this.adjustRow(boxes, rowStart, i - 1, rowHeight, rowWidth, styleSize.width);
                x = paddingLeft;
                y += rowHeight + this.verticalSpacing;
                right = x + size.width;
                rowStart = i;
                rowHeight = 0;
                rowWidth = 0;
            }

            rowHeight = Math.max(rowHeight, size.height);
            rowWidth += size.width;

            //          We are laying out an autoScroll Container, and this item is going to cause a vertical scrollbar:
            //          Adjust the right padding to account for that scrollbar, and reflow.
            if (container.autoScroll && !this.reflow && ((y + rowHeight) > bottomLimit)) {
                right = this.padding.right;
                this.padding.right += Ext.getScrollBarWidth();
                this.reflow = true;
                this.onLayout(container, layoutTarget);
                delete this.reflow;
                this.padding.right = right;
                return this.childBoxCache;
            }

            boxes.push( {
                component : component,
                left : x,
                top : y,
                height : size.height,
                width : size.width
            });
            x = right + this.horizontalSpacing;
        }

        //      Adjust the last row
        this.adjustRow(boxes, rowStart, i - 1, rowHeight, rowWidth, styleSize.width);

        //      Stretch the container heightwise.
        if (!this.sizer) {
            this.sizer = this.innerCt.insertFirst( {
                cls : 'ux-float-layout-sizer'
            });
        }
        this.sizer.setHeight(y + rowHeight - paddingTop);

        return {
            boxes : boxes,
            meta : { tooNarrow : true }
        };
    },

    //  Adjust vertical alignment within row.
    //  Adjust horizontal alignment if required.
    adjustRow : function(boxes, rowStart, rowEnd, rowHeight, rowWidth, availWidth) {
        var i, c, h, j = 0, spareWidth = availWidth - rowWidth, gaps = rowEnd - rowStart, alignmentIncrement = 0;

        switch (this.horizontalAlign) {
        case 'middle':
        case 'center':
            alignmentIncrement = Math.max(spareWidth / 2, 0);
            break;
        case 'right':
            alignmentIncrement = Math.max(spareWidth, 0);
            break;
        case 'justify':
            if (gaps) {
                j = Math.max(spareWidth / gaps, 0);
            }
        }

        for (i = rowStart; i <= rowEnd; i++) {
            c = boxes[i].component;
            h = c.getHeight() + c.getPositionEl().getMargins('tb');
            boxes[i].left += alignmentIncrement;
            alignmentIncrement += j;
            switch (c.verticalAlign || this.verticalAlign) {
            case 'middle':
            case 'center':
                boxes[i].top += (rowHeight - h) / 2;
                break;
            case 'bottom':
                boxes[i].top += (rowHeight - h);
            }
        }
    },
    
    getLayoutTargetSize : function() {
        if (!this.sizer) {
            return Ext.layout.FloatLayout.superclass.getLayoutTargetSize.call(this);
        }
        return this.sizer.getSize();
    }
});
Ext.Container.LAYOUTS['float'] = Ext.layout.FloatLayout;
Ext.layout.boxOverflow.Float = Ext.extend(Ext.layout.boxOverflow.None, {

    handleOverflow : function(cals, tsize) {
        return { 
                targetSize : this.layout.sizer.getSize() 
            };
    }

});