/**
 * Elemento scroll que  utilizada na grid.
 * 
 * @author Patrick Nascimento
 */
Ext.ux.senior.grid.Scroll = Ext.extend(Ext.util.Observable, {

    /**
     * O elemento hmtl da scroll.
     */
    scroll : null,

    /**
     * Listeners para os eventos da scroll.
     */
    listeners : null,

    /**
     * Flag que permite disparar os eventos da scroll. 
     * Necessria para quando algum navegar pelo teclado e setar a posio da scroll, onde neste caso no deve disparar o evento.
     */
    canFireEvent : true,

    /**
     * Utilizado para calcular o quanto a scroll foi movida, e com isto definir se navegou um registro ou uma pgina.
     */
    pageScrollMultiplier : 0.2,

    constructor : function(config) {

        this.scroll = config.element;
        this.listeners = config.listeners || {};

        this.addEvents('beforescroll', 'afterscroll', 'scrolldown', 'scrollup', 'scrollpagedown', 'scrollpageup', 'scrollfirst', 'scrolllast');

        Ext.ux.senior.grid.Scroll.superclass.constructor.call(this, config);

        this.initEvents();
    },

    /**
     * Inicializa os eventos da scroll.
     * @private
     */
    initEvents : function() {
        this.scroll.on("scroll", function() {
            if (this.canFireEvent === true) {
                clearTimeout(this.flag);
                this.flag = this.onScroll.defer(50, this);
            }
            this.canFireEvent = true;

        }, this);

        this.lastScrollbarPos = this.scroll.dom.scrollTop;

    },

    /**
     * Evento de scroll.
     * @private
     */
    onScroll : function(e) {
        var scrollTop = this.getScrollTop();
        var scrollHeight = this.getScrollHeight();
        var fireEvt = null, scrollEvt;

        //se o tamanho do scroll no for maior que zero,  porque ele no esta visivel, ento no  preciso tratar este evento
        if (scrollHeight <= 0) {
            return; // 
        } else if ((scrollTop + this.getHeight() + 5) > scrollHeight) {
            /* aceita at 5px para colar no fim */
            fireEvt = 'scrolllast';
            scrollEvt = ScrollEvent.LAST;
        } else if (scrollTop < 5) {
            /* aceita at 5px para colar no topo */
            fireEvt = 'scrollfirst';
            scrollEvt = ScrollEvent.FIRST;
        } else {
            var dif = this.lastScrollbarPos - scrollTop;
            if (Math.abs(dif) < scrollHeight * this.pageScrollMultiplier) {
                /* moveu pouco, apenas 1 linha. */
                if (dif > 0) {
                    fireEvt = 'scrollup';
                    scrollEvt = ScrollEvent.UP;
                } else {
                    fireEvt = 'scrolldown';
                    scrollEvt = ScrollEvent.DOWN;
                }
            } else {
                if (dif > 0) {
                    fireEvt = 'scrollpageup';
                    scrollEvt = ScrollEvent.PAGEUP;
                } else {
                    fireEvt = 'scrollpagedown';
                    scrollEvt = ScrollEvent.PAGEDOWN;
                }
            }
        }

        if (this.fireEvent('beforescroll', scrollEvt) === false) {
            return;
        }

        this.fireEvent(fireEvt);
        this.fireEvent('afterscroll', scrollEvt);
    },

    /**
     * Define o tamanho da scroll.
     * @param {Number}
     *            height o tamanho.
     * 
     * @public
     */
    setHeight : function(height) {
        this.scroll.setHeight(height);
    },

    /**
     * Retorna o tamanho visvel da scroll.
     * @public
     */
    getHeight : function() {
        return this.scroll.getHeight();
    },

    /**
     * Define a posio da scroll. Altera diretamente a propriedade scrollTop do elemento html.
     * @param {Number}
     *            scrollTop a posio da scroll.
     * @param {Boolean}
     *            fireEvent true para permitir lanar o evento de alterao de posio.
     * 
     * @public
     */
    setScrollTop : function(scrollTop, fireEvent) {
        this.lastScrollbarPos = scrollTop;

        if (this.getScrollTop() === scrollTop) {
            return;
        }

        if (!fireEvent) {
            this.canFireEvent = false;
        }

        this.scroll.dom.scrollTop = scrollTop;
    },

    /**
     * Retorna a posio da scroll, a propriedade scrollTop do elemento html.
     * @public
     */
    getScrollTop : function() {
        return this.scroll.dom.scrollTop;
    },

    /**
     * Retorna a altura total da scroll, a propriedade scrollHeight do elemento html.
     * @private
     */
    getScrollHeight : function() {
        return this.scroll.dom.scrollHeight;
    }

});

/** Esta 'enumerao' segue a ordem da enum VScrollGrid.RequestType. */
ScrollEvent = {
    FIRST : 1,
    PAGEUP : 2,
    UP : 3,
    DOWN : 4,
    PAGEDOWN : 5,
    LAST : 6,
    REFRESH : 7,
    SELECT : 8
};
