/**
 * @author Patrick.Nascimento
 */
Ext.ux.senior.grid.MultiSelectionModel = Ext.extend(Ext.grid.AbstractSelectionModel, {

    /* guarda o ltimo registro focado. */
    lastFocused : -1,

    /**
     * @cfg {Array} initialSelectedRows
     * Linhas que devem aparecer selecionadas inicialmente.
     */

    /**
     * @cfg {Number} initialFocusedRow
     * Linha que deve aparecer focada inicialmente.
     */

    /**
     * @override
     */
    initEvents : function() {
        this.grid.on('rowclick', this.rowClickHandler, this);
        this.grid.on('viewready', this.onViewReady, this, {
            single : true
        });
    },

    /**
     * Dispara o evento de click de uma linha da grid.
     * 
     * @param {Ext.Grid} grid componente que sofreu o click
     * @param {Number} row indice da linha que sofreu o click
     * @param {Event} event evento do browser.
     * @protected
     */
    rowClickHandler : function(grid, row, event) {
        if (row < 0 || row >= this.grid.store.getCount()) {
            return;
        }
        this.fireSelectionEvent("focusRow", "ROW", event, row);
    },

    /**
     * Executa as aes que s podem ser feitas depois que a view da grid j foi renderizada.
     * @private
     */
    onViewReady : function() {
        // Seleciona as linhas iniciais.
        if (this.initialSelectedRows) {
            this.selectRows(this.initialSelectedRows);

            delete this.initialSelectedRows;
        }

        // Foca a linha inicial.
        if (this.initialFocusedRow !== undefined) {
            this.focusRow(this.initialFocusedRow);

            delete this.initialFocusedRow;
        }
    },

    /**
     * Dispara um evento de seleo na grid.<br>
     * Esta seleo pode ser pelo mouse ou por teclas.
     * 
     * 
     * @param {String} eventType 'focusRow' para focar uma linha ou 'selectRow' para selecionar uma linha.
     * @param {String} typeSelection o tipo de seleo (NEXT, PREVIOUS, ROW).
     * @param {Event} event o evento do browser.
     * @param {Number} row a linha que foi selecionada.
     * 
     * @protected
     */
    fireSelectionEvent : function(eventType, typeSelection, evtObject, row) {

        if (this.isLocked()) {
            return;
        }

        var selection = evtObject.ctrlKey ? "MULTI" : "SINGLE";
        selection = evtObject.shiftKey ? "INTERVAL" : selection;

        row = row !== undefined ? row : -1;
        this.grid.fireEvent("selection", this.grid, eventType, typeSelection, selection, row);
    },

    /**
     * Seleciona mltiplas linhas.
     * 
     * @param {Array} rows Array de ndices das linhas a serem selecionadas.
     * @param {Boolean} keepExisting <tt>true</tt> para manter a seleo existente.
     * 
     * @public
     */
    selectRows : function(rows, keepExisting) {
        if (!keepExisting) {
            this.clearSelections();
        }
        for ( var i = 0, len = rows.length; i < len; i++) {
            this.selectRow(rows[i], true);
        }
    },

    /**
     * Seleciona uma linha.
     * 
     * @param {Number} row o ndice da linha a ser selecionada.
     * @param {Boolean} keepExisting <tt>true</tt> para manter a seleo.
     * 
     * @public
     */
    selectRow : function(index, keepExisting) {
        if (this.isLocked() || (index < 0 || index >= this.grid.store.getCount())) {
            return;
        }
        if (!keepExisting) {
            this.clearSelections();
        }
        this.grid.getView().onRowSelect(index);
    },

    /**
     * Deseleciona uma linha.
     * 
     * @param {Number} row o ndice da linha a ser deselecionada.
     * 
     * @public
     */
    deselectRow : function(index) {
        if (this.isLocked()) {
            return;
        }
        this.grid.getView().onRowDeselect(index);
    },

    /**
     * Limpa a seleo das linhas.
     * 
     * @public
     */
    clearSelections : function() {
        if (this.isLocked()) {
            return;
        }

        this.grid.getView().clearSelection();
    },

    /**
     * Seleciona todas as linhas.
     * 
     * @public
     */
    selectAll : function() {
        if (this.isLocked()) {
            return;
        }

        for ( var i = 0, len = this.grid.store.getCount(); i < len; i++) {
            this.selectRow(i, true);
        }
    },

    /**
     * Trata eventos de keydown do GridPanel.
     * 
     * @param {Number} key keyCode da tecla pressionada.
     * @param {Boolean} ctrl <code>true</code> caso a tecla ctrl esteja pressionada.
     * @param {Boolean} shift <code>true</code> caso a tecla shift esteja pressionada.
     * 
     * @protected
     */
    onKeyDown : function(event, key, ctrl, shift) {
        var E = Ext.EventObject;
        var grid = this.grid, view = grid.view;

        var selection = this.lastFocused;
        var firstVisibleRow = view.getFirstRecordIndex();
        var lastVisibleRow = firstVisibleRow + view.visibleRows - 1;
        switch (key) {
        case E.UP:
            var selectRow = selection - 1;
            this.select(selectRow);
            if (ctrl) {
                this.fireSelectionEvent("selectRow", "PREVIOUS", event, selectRow);
            }
            break;
        case E.DOWN:
            var selectRow = selection + 1;
            this.select(selectRow);
            if (ctrl) {
                this.fireSelectionEvent("selectRow", "NEXT", event, selectRow);
            }
            break;

        case E.PAGE_UP:
            if (ctrl) {
                /* Foca o primeiro registro da pgina. */
                view.stopEvents();
                this.selectFirstRecord(grid);
            } else {
                /* se for autoheight no faz nada, s tem uma pgina */
                if (grid.autoHeight) {
                    return;
                }
                /* Navega para a pgina anterior. */
                var selectRow = -1;
                if (selection != null) {
                    if (selection >= firstVisibleRow && selection <= lastVisibleRow) {
                        selectRow = firstVisibleRow - view.visibleRows;
                    } else {
                        selectRow = selection - view.visibleRows;
                    }

                }
                if (grid.getStore().isAtBof() && selectRow < 0) {
                    this.select(0);
                } else {
                    this.select(selectRow);
                    /* Rolou um pageUp, a lgica do pageUp diz que  necessrio subir uma pgina e focar o primeiro registro da pgina.
                     * Para subir uma pgina basta mostrar como primeiro registro visvel a linha que tem o ndice igual  "selectRow"*/
                    view.stopEvents();
                    view.showRecordAtFirst(selectRow);
                    view.resumeEvents();
                }
            }
            Ext.EventObject.stopEvent();
            break;

        case E.PAGE_DOWN:
            if (ctrl) {
                /* Foca o ltimo registro da pgina. */
                view.stopEvents();
                this.selectLastRecord(grid);
            } else {
                /* se for autoheight no faz nada, s tem uma pgina */
                if (grid.autoHeight) {
                    return;
                }
                /* Navega para a prxima pgina. */
                /* A partir deste ndice ser feito o pageDown, ou seja o registro abaixo deste ndice ser o primeiro visvel da nova pgina */
                var indexPageDownRow = -1;
                var selectRow = -1;
                if (selection != null) {
                    if (selection >= firstVisibleRow && selection <= lastVisibleRow) {
                        indexPageDownRow = lastVisibleRow;
                    } else {
                        indexPageDownRow = selection;
                    }
                    selectRow = indexPageDownRow + view.visibleRows;
                }
                if (grid.getStore().isAtEof() && selectRow > grid.getStore().getCount() - 1) {
                    this.select(grid.getStore().getCount() - 1);
                } else {
                    this.select(selectRow);
                    if (indexPageDownRow > -1) {
                        /* Rolou um pageDown, a lgica do pageDown diz que  necessrio descer uma pgina e focar o ltimo registro
                         * para descer uma pgina basta mostrar como primeiro registro visvel a linha aps o "indexPageDownRow"*/
                        view.stopEvents();
                        view.showRecordAtFirst(indexPageDownRow + 1);
                        view.resumeEvents();
                    }
                }
            }
            Ext.EventObject.stopEvent();
            break;

        case E.HOME:
            if (ctrl) {
                if (grid.getStore().isAtBof()) {
                    this.select(0);
                } else {
                    /* Navega e foca o primeiro registro. */
                    view.beforeScroll();
                    view.scrollFirst(true);
                    view.afterScroll(ScrollEvent.FIRST);
                }
            }
            break;

        case E.END:
            if (ctrl) {
                if (grid.getStore().isAtEof()) {
                    this.select(grid.getStore().getCount() - 1);
                } else {
                    /* Navega e foca o ltimo registro. */
                    view.beforeScroll();
                    view.scrollLast(true);
                    view.afterScroll(ScrollEvent.LAST);
                }
            }
            break;

        case E.SPACE:
            /* seleciona a linha focada. */
            if (shift) {
                grid.fireEvent("selection", grid, "selectRow", "ROW", "INTERVAL", this.lastFocused);
            } else {
                grid.fireEvent("selection", grid, "selectRow", "ROW", "MULTI", this.lastFocused);
            }
            break;
        }

        view.resumeEvents();
    },

    select : function(row) {
        if (this.beforeSelection(row)) {
            // this.selectRow(row, false);
            this.focusRow(row, true);
        }
    },

    beforeSelection : function(row) {
        var selection = this.lastFocused;
        if (row < 0 || row > this.grid.getStore().getCount() - 1) {
            selection = -1;
        }
        if (selection < 0) {
            var key = Ext.EventObject.getKey();
            switch (key) {
            case Ext.EventObject.UP:
                this.grid.view.fireScrollEvent(ScrollEvent.UP, -1, false);
                return false;
            case Ext.EventObject.DOWN:
                this.server = true;
                this.grid.view.fireScrollEvent(ScrollEvent.DOWN, -1, false);
                return false;
            case Ext.EventObject.PAGEDOWN:
                if (!Ext.EventObject.ctrlKey) {
                    this.server = true;
                    this.grid.view.fireScrollEvent(ScrollEvent.PAGEDOWN, -1, false);
                    return false;
                }
                break;
            case Ext.EventObject.PAGEUP:
                if (!Ext.EventObject.ctrlKey) {
                    this.server = true;
                    this.grid.view.fireScrollEvent(ScrollEvent.PAGEUP, -1, false);
                    return false;
                }
                break;
            }
        }
        var grid = this.grid;
        var view = grid.view;
        var type;
        if (row < selection) {
            type = ScrollEvent.UP;
        } else {
            type = ScrollEvent.DOWN;
        }
        var key = Ext.EventObject.getKey();
        if (Ext.EventObject.isNavKeyPress()) {
            view.fireScrollEvent(type, row, true);
        }
        return true;
    },

    /**
     * Selectiona o primeiro registro visivel da grid.
     * 
     * @private
     * @param grid referencia da grid para este selection model
     */
    selectFirstRecord : function(grid) {
        this.select(grid.getView().getFirstRecordIndex());
    },

    /**
     * Seleciona o ltimo registro visivel da grid.
     * 
     * @private
     * @param grid
     *            referencia da grid para este selection model
     */
    selectLastRecord : function(grid) {
        var view = grid.getView();
        var selectedRow = view.getFirstRecordIndex() + view.visibleRows - 1;
        /*
         * garante que a ultima linha selecionada seja uma vlida, pois pode ocorrer do store ser menor que o numero da area de linhas "visiveis" da
         * grid
         */
        var storeVisible = this.grid.getStore().getCount() - 1;
        if (selectedRow > storeVisible) {
            selectedRow = storeVisible;
        }
        this.select(selectedRow);
    },

    /**
     * Chama o GridView para adicionar um estilo de focado e focar um registro.<br>
     * O GridView sabe se  necessrio focar no registro ou no.<br>
     * 
     * <b> Funo chamada pelo GridPanel. </b>
     * 
     * @param {Number} index ndice do registro a ser focado.
     * @param {Boolean} forceHtmlFocus <code>true</code> para forar o GridView a setar o foco no elemento html, assim fazendo o registro ficar
     *            visvel.
     * 
     * @protected
     */
    focusRow : function(index, forceHtmlFocus) {
        var view = this.grid.getView();
        view.removeFocusedRowStyle(this.lastFocused);
        if (index > -1) {
            // apenas tenta focar se a Grid for o componente que detm o foco
            if (FocusManager.getFocusedComponent() == this.grid) {
                view.internalFocusRow(index, forceHtmlFocus);
            }
            view.addFocusedRowStyle(index);
        }
        this.lastFocused = index;
    },

    /**
     * Funo utilizada pelas rotinas de drag&drop. Est aqui apenas para no gerar erros.<br>
     * Este SelectionModel no guarda seleo, ento no  vivel implementar.
     * 
     * <b> No utilizar pois sempre retorna um valor padro</b>
     */
    isSelected : function(index) {
        return true;
    },

    /**
     * Funo utilizada pelas rotinas de drag&drop. Est aqui apenas para no gerar erros.<br>
     * Este SelectionModel no guarda seleo, ento no  vivel implementar.
     * 
     * <b> No utilizar pois sempre retorna um valor padro</b>
     */
    getSelections : function() {
        this.grid.store.getAt(0);
    },

    /**
     * Funo utilizada pelas rotinas de drag&drop. Est aqui apenas para no gerar erros.<br>
     * Este SelectionModel no guarda seleo, ento no  vivel implementar, alm do que o servidor no sabe quantos so os registros que esto
     * selecionados.
     * 
     * <b> No utilizar pois sempre retorna um valor padro</b>
     */
    getCount : function() {
        return "";
    },

    /**
     * Funo que verifica se existe alguma clula visvel focada. <br>
     * Apenas  possvel verificar se existe alguma clula visvel focada, pois o client no guarda se existe alguma clula no visvel selecionada.
     */
    hasSelection : function() {
        if(this.grid.el.child(".ux-grid-row-focused")){
            return true;
        }
        return false;
    }

});