Ext.namespace('Ext.ux.senior.grid');

Ext.ux.senior.grid.GridView = function(config) {
    Ext.ux.senior.grid.GridView.superclass.constructor.call(this, config);

    /**
     * Eventos adicionais lanados pelo GridView.
     */
    this.addEvents(
    /**
     * @event beforeviewupdated
     * Evento interno. Disparado antes de uma atualizao de qualquer natureza que ocorreu na grid.
     * @see viewupdated
     * @param {Ext.grid.GridView} view
     * @param rowIndex indice do registro que sofreu a atualizao. Se for omitido ser para toda a grid.
     */
    'beforeviewupdated',
    /**
     * @event viewupdated
     * Evento interno. Disparado quando uma atualizao de qualquer natureza ocorreu na grid.
     * @see beforeviewupdated
     * @param {Ext.grid.GridView} view
     * @param rowIndex indice do registro que sofreu a atualizao. Se for omitido ser para toda a grid.
     */
    'viewupdated', 'sortcolumn', 'realignEditor');
};

Ext
        .extend(
                Ext.ux.senior.grid.GridView,
                Ext.grid.GridView,
                {

                    /**
                     * Define se a linha focada vai ser destacada quando estiver usando o CellSelectionModel.
                     */
                    highlightFocused : 'CELL',

                    /**
                     * Estilo de uma clula selecionada. <br>
                     * S  aplicado caso a grid esteja usando o CellSelectionModel.
                     */
                    selectedCellClass : "ux-grid-cell-selected",

                    /**
                     * Estilo de uma clula selecionada. <br>
                     * S  aplicado caso a grid esteja usando o MultiSelectionModel.
                     */
                    focusedRowClass : "ux-grid-row-focused",

                    /**
                     * Classe css aplicada na clula selecionada
                     */
                    appliedSelectedCellClass : null,

                    /**
                     * Classe css aplicada na linha focada. Utilizado apenas em grid com MultiSelectionModel
                     */
                    appliedFocusedRowClass : null,

                    /**
                     * Classe css aplicada na linha selecionada
                     */
                    appliedSelectedRowClass : [],

                    /**
                     * Tamanho, em pxels, de cada linha da grid.
                     */
                    rowHeight : 22,

                    /* Quantidade mxima de possveis registros visveis na grid. */
                    visibleRows : 0,

                    /* Scroll's elements. */
                    scroll : null,
                    scrollbar : null,
                    scrollWrap : null,

                    /* Flags para controle de envio de requisies de scroll ao servidor. */
                    scrollSent : false,
                    scrollOrder : -1,
                    /* ltima posio da rea de dados. */
                    lastScrollDataPos : 0,

                    /* Flag para controlar os eventos 'scroll' do browser (um no disparar o outro). */
                    stopped : false,

                    /* Controle para saber se o editor deve manter-se ativo aps um refresh. */
                    keepEditing : false,

                    /* Controle para saber se o highlight da grid ser exibido quando ela possui o foco */
                    canShowHighlightFocused : false,

                    /**
                     * @cfg {Number} scrollSizeOffset tamanho da largura da scroll, tanto para scroll vertical ou horizontal.
                     */
                    scrollSizeOffset : 16,

                    /**
                     * Quando uma coluna da grid  aumentada at criar scroll horizontal, o tamanho da grid pode mudar para que a ltima linha no fique pela metade.
                     * Por este motivo,  guardado o tamanho inicial.
                     */
                    initialHeight : 0,

                    /**
                     * Evento que possibilita o drag e drop nas colunas.
                     * @param dataStore
                     * @param columnModel
                     */
                    initData : function(dataStore, columnModel) {
                        Ext.ux.senior.grid.GridView.superclass.initData.call(this, dataStore, columnModel);
                        this.addEvents('realignEditor');
                        if (this.cm) {
                            this.cm.on('columnChangePosition', this.onColumnChangePosition, this);
                        }
                        if(dataStore){
                            dataStore.on('updateEditor', this.updateEditor, this);
                        }
                    },
                    
                    updateEditor : function(view, rowIndex, record) {
                        if (this.grid.activeEditor != null && typeof this.grid.activeEditor !== 'undefined') {
                            this.grid.startEditing(rowIndex,this.grid.activeEditor.col);
                            this.fireEvent("realignEditor", this, record, rowIndex);
                        }

                        //O IE acaba perdendo a edio aps o realignEditor, por isso  necessrio fazer o startEditing novamente, mas com defer.
                        if(Ext.isIE && (this.grid.lastActiveEditor != null && typeof this.grid.lastActiveEditor !== 'undefined')){
                            this.grid.startEditing.defer(2, this.grid, [ rowIndex, this.grid.lastActiveEditor.col ]);
                        }
                    },

                    /**
                     * Cria os elementos da scroll dinmica e adiciona os eventos necessrios para ela funcionar.
                     */
                    onViewReady : function() {
                        if (this.grid.border === false) {
                            this.grid.addClass('ux-grid-noborder');
                        }

                        this.scroller.setStyle('overflow-y', 'hidden');

                        this.scrollWrap = this.el.insertFirst({
                            cls : 'grid-scroll-wrap'
                        });

                        var scroll = this.scrollWrap.insertFirst({
                            cls : 'grid-scroll'
                        });

                        this.scroll = new Ext.ux.senior.grid.Scroll({
                            element : scroll,
                            listeners : {
                                'scrollfirst' : this.scrollFirst,
                                'scrollpageup' : this.scrollPageUp,
                                'scrollup' : this.scrollUp,
                                'scrolldown' : this.scrollDown,
                                'scrollpagedown' : this.scrollPageDown,
                                'scrolllast' : this.scrollLast,
                                'afterscroll' : this.afterScroll,
                                'beforescroll' : this.beforeScroll,
                                scope : this
                            }
                        });

                        this.scrollBt = scroll.insertSibling({
                            cls : 'grid-scroll-bt'
                        }, 'after');

                        this.scrollBt.on("click", function() {
                            this.grid.fireEvent("showFocused");
                        }, this);

                        this.scrollbar = scroll.insertFirst({});
                        this.refreshScroll();

                        this.mainWrap.on('mousewheel', this.handleWheel, this);

                        /* desliga o evento da gridview do Ext pois ns controlamos a scroll. */
                        this.ds.un("load", this.onLoad, this);

                        if (this.grid.border === false) {
                            /* de acordo com analista e seus requisitos, sem borda deixa header transparente. */
                            this.mainHd.addClass("ux-grid-transparent-header");
                        }

                        /* Ajusta o menu de ordenao, remove os menus de esconder coluna */
                        /* remove o iten de colunas */
                        this.hmenu.remove('columns');
                        /* remove o separator, vai estar por ltimo */
                        this.hmenu.remove(this.hmenu.items.last());

                        this.on('beforerefresh', this.onBeforeRefresh);
                        this.on('refresh', this.onRefresh);

                        /* evento de alterao de largura das colunas da grid. */
                        this.grid.on('columnresize', function(columnIndex, newSize) {
                            /* apenas verifica se vai mostrar scroll horizontal (e talvez vertical se mostrar a horizontal). */
                            this.refreshScroll();

                            /*recalcula a posio do scroll pois, caso o scroll vertical esteja no final e o scroll horizontal aparea, o scroll horizontal cobre o ltimo registro*/
                            if (this.scroll && this.isScrollAtBottom() && this.isHorizontalScrollVisible()) {
                                this.stopEvents();
                                this.showRecordAtFirst(this.getFirstRecordIndex() + 1);
                                this.resumeEvents(false);
                            }
                        }, this);

                        /* Ateno: a atualizao dos hints da coluna pode ser feito aqui se os TabPanels utilizar deferredRender=true(default), pois da o clculo
                         * ser feito quando as colunas estiverem visveis, caso contrrio as larguras sero 0 e sempre aparecero os hints.
                         */
                        this.updateHeaderTooltip();
                        this.grid.ownerCt.on('expand', this.refreshScroll, this);
                    },

                    /**
                     * Sobrescrito para adicionar uma template de header com checkbox (UC-FRC-0139 Definir grid - REQ0033)
                     */
                    initTemplates : function() {
                        Ext.ux.senior.grid.GridView.superclass.initTemplates.call(this);

                        var checkhcell = new Ext.Template(
                            '<td class="x-grid3-hd x-grid3-cell x-grid3-td-{id} {css}" style="{style}">',
                                '<div {tooltip} {attr} class="x-grid3-hd-inner x-grid3-hd-{id}" unselectable="on" style="{istyle}">', 
                                    this.grid.enableHdMenu ? '<a class="x-grid3-hd-btn" href="#"></a>' : '',
                                    '<input type="checkbox" id="{id}_checkbox" {checked} {readOnly} class="ux-grid-header-checkbox">&nbsp;',
                                    '{value}',
                                    '<img alt="" class="x-grid3-sort-icon" src="', Ext.BLANK_IMAGE_URL, '" />',
                                '</div>',
                            '</td>'
                        );
                        
                        this.templates.checkhcell = checkhcell;
                    },

                    /**
                     * @override
                     */
                    layout : function() {
                        if (!this.mainBody) {
                            return; // not rendered
                        }

                        var grid = this.grid, gridEl = grid.getGridEl(), gridSize = gridEl.getSize(true), gridWidth = gridSize.width, gridHeight = gridSize.height, scroller = this.scroller, scrollStyle, headerHeight, scrollHeight;

                        if (gridWidth < 20 || gridHeight < 20) {
                            return;
                        }

                        if (grid.autoHeight) {
                            scrollStyle = scroller.dom.style;
                            scrollStyle.overflowX = 'auto';

                            if (Ext.isWebKit) {
                                scrollStyle.position = 'static';
                            }
                        } else {
                            this.el.setSize(gridWidth, gridHeight);

                            headerHeight = this.mainHd.getHeight();
                            scrollHeight = gridHeight - headerHeight;

                            scroller.setSize(gridWidth, scrollHeight);

                        }
                        if (this.innerHd) {
                            this.innerHd.style.width = (gridWidth) + "px";
                        }

                        if (this.forceFit) {
                            if (this.lastViewWidth != gridWidth) {
                                this.fitColumns(false, false);
                                this.lastViewWidth = gridWidth;
                            }
                        } else {
                            this.autoExpand();
                            this.syncHeaderScroll();
                        }

                        this.onLayout(gridWidth, scrollHeight);
                    },

                    /**
                     * @override
                     */
                    renderHeaders : function() {
                        var cells = [];
                        for ( var i = 0; i < this.cm.getColumnCount(); i++) {
                            cells[i] = this.createColumnHeader(i);
                        }

                        return this.templates.header.apply({
                            cells : cells.join(""),
                            tstyle : String.format("width: {0};", this.getTotalWidth())
                        });
                    },

                    /**
                     * @override
                     */
                    handleHdDown : function(e, target) {
                        Ext.ux.senior.grid.GridView.superclass.handleHdDown.call(this, e, target);
                        if (Ext.fly(target).hasClass('ux-grid-header-checkbox')) {
                            var header    = this.findHeaderCell(target);
                            var index = this.getCellIndex(header);
                            if (!this.grid.isReadOnly() && this.cm.config[index].editable) {
                                e.stopPropagation();
                                this.grid.fireEvent("checkBoxClick",index, target.checked);
                            }else{
                                e.stopEvent();
                            }
                        }else{
                            Ext.ux.senior.grid.GridView.superclass.handleHdDown.call(this, e, target);
                        }
                    },

                    /**
                     * Cria o header da coluna <code>columnId</code>
                     * @param columnId id da coluna
                     * @returns html do header da coluna
                     */
                    createColumnHeader : function(columnId) {
                        var properties = {
                            id : this.cm.getColumnId(columnId),
                            value : this.cm.getColumnHeader(columnId) || '',
                            style : this.getColumnStyle(columnId, true),
                            css : this.getColumnStyle(columnId),
                            attr : this.cm.config[columnId].attr || '',
                            checked : this.cm.getHeaderCheckBoxChecked(columnId)? 'checked' : '',
                            readOnly : this.grid.isReadOnly() || !this.cm.config[columnId].editable ? 'readOnly' : '',        
                            tooltip : this.getColumnTooltip(columnId)
                        };

                        if (this.cm.config[columnId].align == 'right') {
                            properties.istyle = 'padding-right: 16px;';
                        } else {
                            delete properties.istyle;
                        }

                        if (this.hasCheckBox(columnId)) { 
                            return this.templates.checkhcell.apply(properties);
                        } else {
                        return this.templates.hcell.apply(properties);
                        }
                    },

                    /**
                     * Retorna a classe css da coluna
                     * 
                     * @param columnId id da coluna
                     * @returns class da coluna
                     */
                    getColumnCss : function(columnId) {
                        var last = this.cm.getColumnCount() - 1;
                        if (i == 0) {
                            return 'x-grid3-cell-first ';
                        } else {
                            return i == last ? 'x-grid3-cell-last ' : '';
                        }
                    },

                    /**
                     * Trata o evento de scroll do mouse (mousewheel) para navegar pelos registros.
                     * 
                     * @private
                     * @param {Ext.EventObject}
                     *            e evento capturado
                     */
                    handleWheel : function(e) {
                        if (!this.isVerticalScrollVisible()) {
                            e.stopEvent();
                            return;
                        }

                        // no  necessrio clicar na grid para scrollar, ento tem que avisar o FocusManager para ela receber o foco.
                        if (FocusManager.getFocusedComponent() !== this.grid) {
                            FocusManager.onComponentFocus(this.grid);
                        }

                        var d = e.getWheelDelta();

                        /* se mantiver editando a posio absoluta do editor vai fazer ele ficar flutuando no local errado */
                        if (this.grid.editing) {
                            this.grid.stopEditing();
                        }

                        if (d < 0) {
                            this.beforeScroll(ScrollEvent.DOWN);
                            this.scrollDown();
                            this.afterScroll(ScrollEvent.DOWN);
                        } else {
                            this.beforeScroll(ScrollEvent.UP);
                            this.scrollUp();
                            this.afterScroll(ScrollEvent.UP);
                        }

                        e.stopEvent();
                    },

                    /**
                     * Mtodo auxiliar para centralizar o acesso  esta varivel.
                     * 
                     * @private
                     */
                    getRowHeight : function() {
                        return this.rowHeight;
                    },

                    /**
                     * Mtodo auxiliar para centralizar o acesso  esta varivel.
                     * 
                     * @private
                     */
                    setRowHeight : function(rowHeight) {
                        this.rowHeight = rowHeight;
                    },

                    /**
                     * Mtodo auxiliar para centralizar o acesso  esta varivel.
                     * 
                     * @private
                     */
                    setScrollDataTop : function(top) {
                        /* valores maiores do que o permitido ou menores que zero so 'arredondados' pelo browser, ou seja, para salvar o last, busca do dom. */
                        this.scroller.dom.scrollTop = top;
                        this.lastScrollDataPos = this.scroller.dom.scrollTop;
                    },

                    /**
                     * Mtodo auxiliar para centralizar o acesso  esta varivel.
                     * 
                     * @private
                     */
                    getScrollDataTop : function() {
                        return this.scroller.dom.scrollTop;
                    },

                    /**
                     * Verifica se, por uma mudana do store,  necessrio mostrar o esconder a scroll.<br>
                     * Caso a mudana do store impacte na scroll, verifica se vai mostrar ou escod-la.
                     * 
                     * @private
                     */
                    verifyNeedShowScroll : function() {
                        var width = this.grid.getWidth();
                        if (this.isVerticalScrollVisible() && !this.grid.autoHeight) {
                            this.scroller.setWidth(width - this.getScrollOffset());
                            this.scrollWrap.setStyle("display", "");
                        } else {
                            this.scroller.setWidth(width);
                            this.scrollWrap.setStyle("display", "none");
                        }
                    },

                    /**
                     * Retorna se deve mostrar a scroll horizontal na grid.<br>
                     * Considera se existe a scroll vertical para mostrar a horizontal.
                     */
                    isHorizontalScrollVisible : function() {
                        var size = this.grid.getGridEl().getSize();
                        var width = size.width;

                        if (!this.forceFit && this.isVerticalScrollVisible()) {
                            return this.cm.getTotalWidth() >= width - this.scroll.scroll.dom.offsetWidth - 2;
                        } else {
                            return this.cm.getTotalWidth() > width;
                        }
                    },

                    /**
                     * Retorna se deve mostrar a scroll vertical na grid.<br>
                     * No considera a scroll horizontal pois se ela estiver sendo exibida, o valor da visibleRows estar alterado e 
                     * isso j faz o clculo ficar correto.
                     */
                    isVerticalScrollVisible : function() {
                        return this.ds.getCount() - this.visibleRows > 0;
                    },

                    /**
                     * Calcula a quantidade de linhas que a grid est mostrando.<br>
                     * Caso o tamanho da grid mude por algum motivo, esta funo deve ser chamada.
                     * 
                     * @private
                     * @returns true se alterou a altura da grid. false se a altura ficou a original.
                     */
                    calculateVisibleRows : function() {
                        var height = this.calculateScrollerHeight();
                        // Desconsidera o tamanho do scroll horizontal, caso ele estiver visvel
                        if (this.isHorizontalScrollVisible()) {
                            height -= this.scrollSizeOffset;
                        }

                        this.visibleRows = Math.max(1, Math.floor(height / this.getRowHeight()));
                        return this.visibleRows;
                    },

                    /**
                     * Ajusta o tamanho do scroller (div com as linhas).
                     */
                    adjustScrollerHeight : function() {
                        if (!this.grid.autoHeight) {
                            var height = this.calculateScrollerHeight();
                            this.scroller.setHeight(height);
                        }
                    },

                    /**
                     * Calcula o tamanho do scroller (viewport dos registros).
                     * 
                     * <pre>
                     * Passos:
                     *  - Calcular altura do scroller (Viewport dos registros)
                     *      + Obtm altura da grid (Desconsiderando bordas)
                     *          - Desconsidera altura do mainHd(barra das colunas)
                     *          - Desconsidera altura do header(barra de ttulo), se tiver
                     * </pre>
                     * 
                     * @returns o tamanho do scroller (viewport dos registros)
                     */
                    calculateScrollerHeight : function() {
                        /* Se no houver uma altura definida no h linhas visveis */
                        if (this.getRowHeight() == 0) {
                            return 0;
                        }

                        /* Verifica a altura do elemento scroller na rvore DOM. Se for 0, a grid pode estar dentro de um painel minimizado. */
                        if (this.scroller.getHeight() == 0) {
                            return 0;
                        }

                        /* Obtem altura da grid desconsiderando suas bordas*/
                        var height = this.getGridHeight(true);

                        /* Desconsidera a altura da barra das colunas considerando suas bordas*/
                        height -= this.mainHd.getHeight();

                        /* Desconsidera a altura da barra do ttulo considerando suas bordas */
                        if (this.grid.header) {
                            height -= this.grid.header.getHeight();
                        }

                        return height;
                    },

                    /**
                     * Retorna a altura da grid desta view.
                     * 
                     * @param contentHeight (optional) true para obter a altura menos as bordas e padding 
                     * @returns altura da grid deste view
                     */
                    getGridHeight : function(contentHeight) {
                        /* Obtem altura da grid */
                        var height = this.grid.height;

                        /* Se for autoHeight ignora a altura definida, ou se no tiver altura definida, pede pelo mtodo para calcular. */
                        if (this.grid.autoHeight || !height) {
                            height = this.grid.getHeight();
                        }

                        if (contentHeight) {
                            /* Desconsidera bordas/padding do body grid */
                            height -= this.grid.body.getBorderWidth('tb') - this.grid.body.getPadding('tb');
                            /* Desconsidera bordas/padding dos filhos do body */
                            var child = this.grid.body.first();
                            while (child) {
                                height -= child.getBorderWidth('tb') - child.getPadding('tb');
                                child = this.grid.body.next();
                            }
                        }

                        return height;
                    },

                    /**
                     * Workarround para ignorar 2 possveis pixels referentes  bordas.<br>
                     */
                    isEquals : function(a, b) {
                        if (a === b) {
                            return true;
                        }
                        if ((Math.max(a, b) - 2) <= Math.min(a, b)) {
                            return true;
                        }
                        return false;
                    },

                    onDataNavigate : function() {
                        var top = this.getScrollDataTop();

                        if (this.isEquals(top, this.lastScrollDataPos)) {
                            return; /* top no mudou, nada faz. */
                        }

                        if (this.stopped) {
                            return;
                        }
                        this.stopEvents();
                        var firstRecordIndex = this.getFirstRecordIndex();
                        var rows = Math.ceil(top / this.getRowHeight()) + this.visibleRows;
                        var scrollBarTop = Math.floor(this.scrollbar.getHeight() * 0.33);

                        if (top > this.lastScrollDataPos) {
                            if (this.ds.getCount() == rows && this.grid.getStore().isAtEof()) {
                                scrollEvent = ScrollEvent.LAST;
                                scrollBarTop = this.scrollbar.getHeight() - this.scroll.getHeight();
                            } else {
                                scrollEvent = ScrollEvent.DOWN;
                            }
                            this.refreshIndexFirstRecord(firstRecordIndex);
                        } else {
                            if (top < 2 && this.grid.getStore().isAtBof()) { /* ignora 1px por causa de borda */
                                scrollBarTop = 0;
                                scrollEvent = ScrollEvent.FIRST;
                            } else {
                                scrollEvent = ScrollEvent.UP;
                            }
                            this.refreshIndexFirstRecord(firstRecordIndex);
                        }

                        this.scroll.setScrollTop(scrollBarTop);

                        this.lastScrollDataPos = top;

                        this.resumeEvents();
                    },

                    /**
                     * Atualiza a flag que guarda o primeiro registro.
                     * 
                     * @private
                     */
                    refreshIndexFirstRecord : function(index) {
                        this.curRec = this.ds.getAt(index);
                    },

                    /**
                     * Pausa os listeners de eventos de scroll.
                     * 
                     * @public
                     */
                    stopEvents : function() {
                        this.stopped = true;
                    },

                    /**
                     * Faz os listeners de eventos de scroll continuarem seu trabalho normalmente.
                     * 
                     * @public
                     */
                    resumeEvents : function(defered) {
                        if (!defered) {
                            return this.resumeEvents.defer(20, this, [ true ]);
                        }
                        this.stopped = false;
                    },

                    /**
                     * 
                     * @private
                     */
                    beforeScroll : function(scrollEvent) {
                        this.stopEvents();
                    },

                    /**
                     * @private
                     */
                    afterScroll : function(scrollEvent) {
                        this.curRec = null;
                        if (scrollEvent != ScrollEvent.LAST && scrollEvent != ScrollEvent.FIRST) {
                            this.refreshIndexFirstRecord(this.getFirstRecordIndex());
                        }
                        this.resumeEvents();
                    },

                    /**
                     * Posiciona a scroll no primeiro registro e envia uma requisio ao servidor.
                     * 
                     * @private
                     */
                    scrollFirst : function(selectFirst, avoidEvent) {
                        if (!avoidEvent) {
                            this.fireScrollEvent(ScrollEvent.FIRST, null, selectFirst);
                        }
                    },

                    /**
                     * Posiciona a scroll uma pgina acima e envia uma requisio ao servidor.
                     * 
                     * @private
                     */
                    scrollPageUp : function(select, avoidEvent) {
                        if (!avoidEvent) {
                            this.fireScrollEvent(ScrollEvent.PAGEUP, null, select);
                        }
                    },

                    /**
                     * Posiciona a scroll um registro acima e envia uma requisio ao servidor.
                     * 
                     * @private
                     */
                    scrollUp : function(avoidEvent) {
                        if (!avoidEvent) {
                            this.fireScrollEvent(ScrollEvent.UP, null, false);
                        }
                    },

                    /**
                     * Posiciona a scroll um registro abaixo e envia uma requisio ao servidor.
                     * 
                     * @private
                     */
                    scrollDown : function(avoidEvent) {
                        if (!avoidEvent) {
                            this.fireScrollEvent(ScrollEvent.DOWN, null);
                        }
                    },

                    /**
                     * Posiciona a scroll uma pgina abaixo e envia uma requisio ao servidor.
                     * 
                     * @private
                     */
                    scrollPageDown : function(select, avoidEvent) {
                        if (!avoidEvent) {
                            this.fireScrollEvent(ScrollEvent.PAGEDOWN, null, select);
                        }
                    },

                    /**
                     * Posiciona a scroll no ltimo registro e envia uma requisio ao servidor.
                     * 
                     * @private
                     */
                    scrollLast : function(selectLast, avoidEvent) {
                        if (!avoidEvent) {
                            this.fireScrollEvent(ScrollEvent.LAST, null, selectLast);
                        }
                    },

                    /**
                     * Envia ao servidor uma requisio avisando que um evento de scroll ocorreu.
                     * 
                     * @param {ScrollEvent}
                     *            event tipo de evento de scroll.
                     * @param {Boolean}
                     *            select <true> para selecionar o primeiro ou ltimo registro em caso de scroll first ou last.
                     * 
                     * @private
                     */
                    fireScrollEvent : function(event, focused, select) {
                        if (this.ds.getCount() > 0) {
                            var firstRecordIndex = this.getFirstRecordIndex();

                            switch (event) {
                            case ScrollEvent.FIRST:
                                this.grid.getStore().setBof(true);
                                firstRecordIndex = 0;
                                this.showRecordAtFirst(firstRecordIndex);
                                break;
                            case ScrollEvent.PAGEUP:
                                var first = Math.max(0, this.getFirstRecordIndex() - this.visibleRows);
                                firstRecordIndex = first;
                                if (!focused) {
                                    this.showRecordAtFirst(firstRecordIndex);
                                }
                                break;
                            case ScrollEvent.UP:
                                var first = Math.max(0, this.getFirstRecordIndex() - 1);
                                if (!focused) {
                                    this.showRecordAtFirst(first);
                                    firstRecordIndex = first;
                                } else {
                                    if (focused < this.getFirstRecordIndex()) {
                                        firstRecordIndex = first;
                                    }
                                }
                                break;
                            case ScrollEvent.DOWN:
                                var limitFirst = Math.max(0, this.ds.getCount() - this.visibleRows);
                                var first = Math.min(limitFirst, this.getFirstRecordIndex() + 1);
                                if (!focused) {
                                    firstRecordIndex = first;
                                    this.showRecordAtFirst(firstRecordIndex);
                                } else {
                                    if (focused >= this.getFirstRecordIndex() + this.visibleRows) {
                                        firstRecordIndex = first;
                                    }
                                }
                                break;
                            case ScrollEvent.PAGEDOWN:
                                var limitFirst = Math.max(0, this.ds.getCount() - this.visibleRows);
                                var first = Math.min(limitFirst, this.getFirstRecordIndex() + this.visibleRows);
                                firstRecordIndex = first;
                                if (!focused) {
                                    this.showRecordAtFirst(firstRecordIndex);
                                }
                                break;
                            case ScrollEvent.LAST:
                                this.grid.getStore().setEof(true);
                                var firstIndex = this.ds.getCount() - this.visibleRows;
                                firstRecordIndex = firstIndex <= 0 ? 0 : firstIndex;
                                this.showRecordAtFirst(firstRecordIndex);
                                break;
                            }

                            this.curRec = this.ds.getAt(firstRecordIndex);
                            var focusedKey = -1;
                            var recordKey = this.curRec.data.recordKey;
                            if (focused == -1) {
                                recordKey = -1;
                            } else {
                                if (select && focused != null) {
                                    focusedKey = this.ds.getAt(focused).data.recordKey;
                                }
                            }
                            this.grid.fireEvent("scroll", recordKey, focusedKey, this.visibleRows, event, !!select);
                        }
                    },

                    /**
                     * Retorna o ndice da primeira linha visvel na grid
                     * 
                     * @return o ndice da primeira linha visvel na grid
                     * 
                     * @public
                     */
                    getFirstRecordIndex : function() {
                        return Math.ceil(this.getScrollDataTop() / this.getRowHeight());
                    },

                    /**
                     * Tem quase o mesmo comportamento que a funo showRecordAtFirst.<br>
                     * O que diferencia  que caso o registro j esteja aparecendo ao usurio, a posio da scroll da grid no  alterada, porm nenhuma flag da grid  modificada, ou seja, assume-se que o registro
                     * deste ndice j esteja selecionado e aparecendo na tela.
                     * 
                     * 
                     * <b>Antes de utilizar esta funo  necessrio pausar os eventos de scroll, e logo aps continar os eventos, usando as funes <code>stopEvents()</code> e <code>resumeEvents</code>.
                     * 
                     * @param {Number}
                     *            recIndex indce do registro baseado no store da grid.
                     * @public
                     */
                    showRecord : function(recIndex) {

                        /* se tem alguma configurao no aplicada ainda, ignora o recIndex que veio e aplica o que estava configurado. */
                        if (this.grid.initialVisibleRow) {
                            recIndex = this.grid.initialVisibleRow;
                            delete this.grid.initialVisibleRow;
                        }

                        var isVisible = recIndex >= this.getFirstRecordIndex() && recIndex < (this.getFirstRecordIndex() + this.visibleRows);

                        if (!isVisible) {
                            this.showRecordAtFirst(recIndex);
                        }
                    },

                    /**
                     * Mostra determinado registro com o primeiro vsivel.<br>
                     * Caso este registro esteje entre os ltimos, mostra a ltima pgina da grid.<br>
                     * 
                     * Tambm ajusta a posio da scroll caso necessrio.<br>
                     * 
                     * <b>Antes de utilizar esta funo  necessrio pausar os eventos de scroll, e logo aps continar os eventos, usando as funes <code>stopEvents()</code> e <code>resumeEvents</code>.
                     * 
                     * @param {Number}
                     *            recIndex indce do registro baseado no store da grid.
                     * @public
                     */
                    showRecordAtFirst : function(recIndex) {
                        if (!this.stopped) {
                            return;
                        }

                        var dataTop, scrollTop = Math.floor(this.scrollbar.getHeight() * 0.33);

                        if (recIndex < 1) {

                            /* posiciona tudo no topo. */
                            dataTop = 0;
                            if (this.grid.getStore().isAtBof()) {
                                scrollTop = 0;
                            }
                        } else {
                            var storeLenght = this.ds.getCount();

                            if (recIndex + this.visibleRows >= storeLenght) {

                                if (this.grid.getStore().isAtEof()) {
                                    /* posiciona tudo no final */
                                    scrollTop = this.scrollbar.getHeight() - this.scroll.getHeight();
                                }
                                /* tem calculo errado (tamanho na tela no  o mesmo do que o calculado), joga valor maior que o browser ignora o resto. */
                                dataTop = (storeLenght - this.visibleRows) * this.getRowHeight() + 2000;
                            } else {
                                /* posiciona scroll bar no meio e scroller de acordo com o recIndex */
                                dataTop = recIndex * this.getRowHeight();
                            }

                        }
                        this.scroll.setScrollTop(scrollTop);
                        this.setScrollDataTop(dataTop);

                    },

                    /**
                     * @override
                     */
                    onUpdate : function(ds, record) {
                        this.fireEvent("beforeviewupdated", this, record);
                        Ext.ux.senior.grid.GridView.superclass.onUpdate.call(this, ds, record);
                        this.refreshScroll();
                        this.fireEvent("viewupdated", this, record);
                    },

                    /**
                     * @override
                     */
                    onAdd : function(ds, records, index) {
                        this.fireEvent("beforeviewupdated", this, records);
                        Ext.ux.senior.grid.GridView.superclass.onAdd.call(this, ds, records, index);
                        this.refreshScroll();
                        if (this.scroll && this.isScrollAtBottom()) {
                            this.stopEvents();
                            this.showRecordAtFirst(this.getFirstRecordIndex());
                            this.resumeEvents(false);
                        }
                        this.fireEvent("viewupdated", this, records);
                    },

                    /**
                     * Verifica se o scroll est no final, apresentando a ltima pgina de registros da grid.
                     * 
                     * Antes de chamar este mtodo, deve-se garantir que o scroll existe. 
                     * 
                     * @returns true se o scroll estiver posicionado no final da barra de scroll.
                     */
                    isScrollAtBottom : function() {
                        return this.scroll.getScrollTop() == (this.scrollbar.getHeight() - this.scroll.getHeight());
                    },

                    /**
                     * Verifica se o scroll est no centro, apresentando um pgina que no seja nem a primeira nem a ltima pgina de registros da grid.
                     * 
                     * Antes de chamar este mtodo, deve-se garantir que o scroll existe. 
                     * 
                     * @returns true se o scroll estiver posicionado no centro da barra de scroll.
                     */
                    isScrollAtCenter : function() {
                        return this.scroll.getScrollTop() == Math.floor(this.scrollbar.getHeight() * 0.33);
                    },

                    /**
                     * @override
                     */
                    onRemove : function(ds, record, index, isUpdate) {
                        if (!isUpdate) {
                            this.fireEvent("beforeviewupdated", this, record);
                        }
                        Ext.ux.senior.grid.GridView.superclass.onRemove.call(this, ds, record, index, isUpdate);
                        this.refreshScroll();
                        if (this.scroll && this.isScrollAtCenter()) {
                            this.stopEvents();
                            this.showRecordAtFirst(this.getFirstRecordIndex());
                            this.resumeEvents(false);
                        }
                        if (!isUpdate) {
                            this.fireEvent("viewupdated", this, record);
                        }
                    },

                    /**
                     * Listener chamado pelo store quando os registros so alterados. <br>
                     * 
                     *  responsvel por atualizar a scroll e garantir que os dados que estavam visveis continuem visveis caso ainda estejam no store.
                     * 
                     * @override.
                     */
                    onDataChange : function() {
                        this.fireEvent("beforeviewupdated", this);
                        this.stopEvents();

                        var scrollState = this.getScrollState();
                        /* superclass redesenha toda os dados. */
                        Ext.ux.senior.grid.GridView.superclass.onDataChange.call(this);

                        if (!this.stopped) {
                            return;
                        }

                        /* sempre verifica pois a grid pode fechar (painel dela), etc.. */
                        if (this.getRowHeight() == -1) {
                            /* se no possui linha ou a altura da grid j est calculada corretamente */
                            if (!this.getRows()[0] || this.getRows()[0].offsetHeight === 0) {
                                return;
                            } else {
                                this.setRowHeight(this.getRows()[0].offsetHeight);

                                this.calculateVisibleRows();
                                this.adjustScrollerHeight();

                                if (this.getRowHeight() <= 0) {
                                    this.setRowHeight(-1);
                                    return;
                                }
                            }
                        }

                        this.refreshScroll();

                        /* depois de atualizar os dados no client, recupera o registro que era o primeiro vsivel e garante que ele continue sendo o primeiro. */
                        if (this.curRec != null) {
                            var index = this.ds.indexOfId(this.curRec.id);
                            if (index >= 0) {
                                this.showRecordAtFirst(index);
                            }
                            this.curRec = null;
                        } else {
                            /* no est mais no client o registro, ento restaura a scroll */
                            this.restoreScroll(scrollState);
                        }

                        this.resumeEvents();

                        this.fireEvent("viewupdated", this);
                    },

                    /**
                     * Define qual o estilo de highlight de registro focado na grid. CELL para exibir apenas a clula focada ROW para exibir alm da clula, a linha focada NONE para no exibir nem a linha nem a
                     * clula focada.
                     * 
                     * @param highligh
                     *            estilo de highlight (CELL, ROW, NONE)
                     * 
                     * @public
                     */
                    setHighlightFocused : function(highlight) {
                        this.highlightFocused = highlight;
                    },

                    /**
                     * Fora a atualizao do estilo de highlight de registro focado na grid.
                     * 
                     * @public
                     */
                    updateHighlightFocused : function() {
                        var selection = this.grid.getSelectionModel().getSelectedCell ? this.grid.getSelectionModel().getSelectedCell() : null;
                        if (!selection || selection.length === 0) {
                            return;
                        }
                        if (this.highlightFocused === 'NONE') {
                            this.removeSelectedRowClass(selection[0]);
                            var cell = this.getCell(selection[0], selection[1]);
                            if (cell) {
                                this.grid.restoreCustomStyle(selection[0], selection[1]);
                                this.removeSelectedCellClass(cell);
                            }
                        }
                        if (this.highlightFocused === 'ROW') {
                            this.removeSelectedRowClass(selection[0]);
                            this.addSelectedRowClass(selection[0]);
                            var cell = this.getCell(selection[0], selection[1]);
                            if (cell) {
                                this.removeSelectedCellClass(cell);
                                this.addSelectedCellClass(cell, selection[0], selection[1]);
                            }
                            
                        }

                        if (this.highlightFocused === 'CELL') {
                            this.removeSelectedRowClass(selection[0]);
                            var cell = this.getCell(selection[0], selection[1]);
                            if (cell) {
                                this.grid.removeCustomStyle(selection[0], selection[1]);
                                this.removeSelectedCellClass(cell);
                                this.addSelectedCellClass(cell, selection[0], selection[1]);
                            }
                        }
                    },

                    /**
                     * Adiciona um estilo de seleo na linha e um estilo de seleo de clula na coluna.<br>
                     *  chamado pelo CellSelecionModel.
                     * 
                     * @override.
                     */
                    onCellSelect : function(row, col) {
                        if (this.canShowHighlightFocused || this.grid.alwaysHighlightFocused) {
                            if (this.highlightFocused === 'ROW') {
                                this.addSelectedRowClass(row);
                            }
                            if (this.highlightFocused !== 'NONE') {
                                var cell = this.getCell(row, col);
                                if (cell) {
                                    this.addSelectedCellClass(cell, row, col);
                                    this.grid.removeCustomStyle(row, col);
                                }
                            }
                        }
                    },

                    /**
                     * Remove o estilo de seleo na linha e o estilo de seleo de clula na coluna.<br>
                     *  chamado pelo CellSelecionModel.
                     * 
                     * @override.
                     */
                    onCellDeselect : function(row, col) {
                        if (this.highlightFocused === 'ROW') {
                            this.removeSelectedRowClass(row);
                        }

                        if (this.highlightFocused !== 'NONE') {
                            var cell = this.getCell(row, col);
                            if (cell) {
                                this.removeSelectedCellClass(cell);
                                this.grid.restoreCustomStyle(row, col);

                            }
                        }
                    },

                    /**
                     * Sobrescrito pois toda vez que  adicionado uma classe de estilo na linha, 
                     *  necessrio remover a classe de estilo customizado que est aplicada na linha.
                     * 
                     * @override
                     */
                    addRowClass : function(row, cls) {
                        Ext.ux.senior.grid.GridView.superclass.addRowClass.call(this, row, cls);
                        if (this.getRow(row)) {
                            this.grid.removeCustomStyle(row);
                        }
                    },

                    /**
                     * Adiciona a classe de estilo de linha selecionada na linha informada.
                     * <br>
                     * Se houver uma classe de estilo customizado de linha selecionada para esta linha, ento esta classe ser aplicada.
                     * Caso contrrio ser aplicada a classe padro 'x-grid-row-selected'
                     * @param row ndice da nova linha selecionada
                     */
                    addSelectedRowClass : function(row) {
                        var cls = null;

                        // verifica se existe um estilo de selecionado customizado para esta linha
                        if (this.grid.focusCustomizations && this.grid.focusCustomizations.selectedCustomizations) {
                            var focusCustomizations = this.grid.focusCustomizations.selectedCustomizations[this.grid.getStore().getAt(row).id];

                            if (focusCustomizations && focusCustomizations.rowCustomization) {
                                cls = focusCustomizations.rowCustomization.style;
                            }
                        }

                        // se no encontrou estilo customizado, ento aplica a classe padro
                        if (!cls) {
                            cls = this.selectedRowClass;
                        }

                        this.appliedSelectedRowClass[row] = cls;
                        this.addRowClass(row, cls);
                    },

                    /**
                     * Remove o estilo de linha selecionada da linha informada
                     * @param row ndice da linha
                     * @param ignoreCustomStyle <code>true</code> para no reaplicar o estilo customizado, caso a linha tenha um, <code>false</code> para reaplicar 
                     */
                    removeSelectedRowClass : function(row, ignoreCustomStyle) {
                        this.removeRowClass(row, this.appliedSelectedRowClass[row], ignoreCustomStyle);
                        this.removeRowClass(row, this.selectedRowClass, ignoreCustomStyle);
                    },

                    /**
                     * Adiciona a classe de estilo de clula selecionada na clula informada.
                     * <br>
                     * Se houver uma classe de estilo customizado de clula selecionada para esta clula, ento esta classe ser aplicada.
                     * Caso contrrio ser aplicada a classe padro 'ux-grid-cell-selected'
                     * 
                     * @param cell o elemento que representa a clula,  onde a classe ser adicionada.
                     * @param row ndice da linha onde se encontra a clula
                     * @param col ndice da coluna onde se encontra a clula
                     */
                    addSelectedCellClass : function(cell, row, col) {
                        var cls = null;

                        // verifica se existe um estilo de selecionado customizado para esta clula
                        if (this.grid.focusCustomizations && this.grid.focusCustomizations.selectedCustomizations) {
                            var focusCustomizations = this.grid.focusCustomizations.selectedCustomizations[this.grid.getStore().getAt(row).id];
                            if (focusCustomizations && focusCustomizations.cellCustomizations) {
                                var cellCustomization = focusCustomizations.cellCustomizations;
                                var fieldName = this.grid.getColumnModel().getDataIndex(col);
                                if (cellCustomization[fieldName]) {
                                    cls = cellCustomization[fieldName].style;
                                }
                            }
                        }

                        // se no encontrou estilo de selecionado customizado, ento aplica a classe padro
                        if (!cls) {
                            cls = this.selectedCellClass;
                        }

                        this.appliedSelectedCellClass = cls;

                        Ext.fly(cell).addClass(cls);
                    },

                    /**
                     * Remove o estilo de clula selecionada da clula informada
                     * @param cell o elemento que representa a clula
                     */
                    removeSelectedCellClass : function(cell) {
                        Ext.fly(cell).removeClass(this.appliedSelectedCellClass);
                        Ext.fly(cell).removeClass(this.selectedCellClass);
                    },

                    /**
                     * Sobrescrito pois pode haver uma classe de estilo customizado para linha selecionada. 
                     * @override
                     */
                    onRowSelect : function(row) {
                        this.addSelectedRowClass(row);
                        // na grid multiselect o estilo de linha selecionada padro tambm  aplicado pois o chekbox de seleo  uma imagem e 
                        //essa imagem  posicionada com um seletor css que utiliza a classe de linha seleciondada
                        this.addRowClass(row, this.selectedRowClass);
                    },

                    /**
                     * Sobrescrito pois pode haver uma classe de estilo customizado para linha selecionada. 
                     * @override
                     */
                    onRowDeselect : function(row) {
                        this.removeSelectedRowClass(row, this.rowHasClass(row, this.appliedFocusedRowClass));
                    },

                    /**
                     * @param ignoreCustomStyle <code>true</code> se no for necessrio reaplicar as classes de estilo customizado que estavam aplicadas na linha.
                     * 
                     */
                    removeRowClass : function(row, cls, ignoreCustomStyle) {
                        Ext.ux.senior.grid.GridView.superclass.removeRowClass.call(this, row, cls);
                        if (!ignoreCustomStyle) {
                            if (this.getRow(row)) {
                                this.grid.restoreCustomStyle(row);
                            }
                        }
                    },

                    /**
                     * Limpa a seleo das linhas.<br>
                     * Chamado pelo SelectionModel.
                     * 
                     * @private
                     */
                    clearSelection : function() {
                        for ( var i = 0; i < this.ds.getCount(); i++) {
                            this.onRowDeselect(i);
                        }
                    },

                    /**
                     * Adiciona a classe de estilo de linha focada na linha informada.
                     * <br>
                     * Se houver uma classe de estilo customizado de linha focada para esta linha, ento esta classe ser aplicada.
                     * Caso contrrio ser aplicada a classe padro 'ux-grid-row-focused'
                     * @param row ndice da nova linha focada
                     */
                    addFocusedRowStyle : function(row) {
                        var cls = this.focusedRowClass;

                        if (this.grid.focusCustomizations && this.grid.focusCustomizations.focusedCustomizations) {
                            var focusCustomizations = this.grid.focusCustomizations.focusedCustomizations;
                            var customization = focusCustomizations[this.grid.getStore().getAt(row).id];
                            if (customization && customization.rowCustomization) {
                                cls = customization.rowCustomization.style;
                            }
                        }

                        if (!cls) {
                            cls = this.focusedRowClass;
                        }

                        this.appliedFocusedRowClass = cls;
                        this.addRowClass(row, cls);

                    },

                    /**
                     * Remove o estilo de linha focada da linha informada
                     */
                    removeFocusedRowStyle : function(row) {
                        this.removeRowClass(row, this.appliedFocusedRowClass, this.rowHasClass(row, this.appliedSelectedRowClass[row]));
                        this.removeRowClass(row, this.focusedRowClass, this.rowHasClass(row, this.appliedSelectedRowClass[row]));
                    },

                    /**
                     * Retorna se a linha possui a classe css que foi informada 
                     */
                    rowHasClass : function(row, cls) {
                        if (this.getRow(row)) {
                            if (this.fly(this.getRow(row)).hasClass(cls)) {
                                return true;
                            }
                        }
                        return false;
                    },

                    /**
                     * 
                     * @public
                     */
                    onColumnChangePosition : function(dataIndex, newIndex) {
                        this.grid.fireEvent("columnChangePosition", dataIndex, newIndex);
                    },

                    /**
                     * 
                     * @private
                     */
                    internalFocusRow : function(row, forceHtmlFocus) {
                        var isVisible = row >= this.getFirstRecordIndex() - 1 && row <= (this.getFirstRecordIndex() + this.visibleRows + 1);

                        if (forceHtmlFocus || isVisible) {
                            Ext.ux.senior.grid.GridView.superclass.focusRow.call(this, row);
                        }
                    },


                    /**
                     * Depois que os headers das colunas so renderizados,  necessrio colocar os cones de sort dos campos de ordenao.
                     */
                    updateHeaders : function() {
                        Ext.ux.senior.grid.GridView.superclass.updateHeaders.call(this);
                        this.renderSortIcons();
                    },

                    /**
                     * Define os campos de ordenao e atualiza os cones de sort dos headers das colunas da grid.
                     */
                    updateSortFields : function(sortFields) {
                        /* Atualiza os sortFields. */
                        this.sortFields = sortFields;

                        /* Remove o cone de sort de todas as colunas. */
                        for ( var i = 0, l = this.cm.getColumnCount(); i < l; i++) {
                            /* Obtm o element do header. */
                            var headerElement = Ext.get(this.getHeaderCell(i));

                            /* Remove os dois possveis cones. */
                            headerElement.removeClass(this.sortClasses);
                        }

                        /* Renderiza os novos cones. */
                        this.renderSortIcons();
                    },

                    /**
                     * Verifica se o texto dos headers das colunas  maior que a largura, se for redesenha os headers com Tooltip. Deve ser chamado aps a grid
                     * renderizar e sempre que mudar o texto ou tamanho de uma coluna.
                     * 
                     * @param col
                     *            A coluna que se deseja atualizar o ndice, se no informado atualiza em todas.
                     */
                    updateHeaderTooltip : function(col) {
                        var v = this;
                        var updateTooltip = function(col) {
                            var header = new Ext.Element(v.getHeaderCell(col));
                            if (header.getTextWidth() > header.getWidth(true)) {
                                v.cm.setColumnTooltip(col, v.cm.config[col].getHeaderTooltip());
                            } else {
                                v.cm.setColumnTooltip(col, null);
                            }
                        };
                        /* atualiza os ttulos, para colocar hint onde no couber o texto */
                        if (col && Ext.isNumber(col)) {
                            updateTooltip(col);
                        } else {
                            for ( var i = 0; i < this.cm.getColumnCount(); i++) {
                                updateTooltip(i);
                            }
                        }
                        this.updateHeaders();
                    },

                    /**
                     * Renderiza os cones de sort dos headers das colunas da grid.
                     */
                    renderSortIcons : function() {
                        var sortFields = this.sortFields;
                        if (!this.sortFields) {
                            return;
                        }

                        /* Adiciona o cone de sort nos campos de ordenao. */
                        var cm = this.grid.getColumnModel();
                        for ( var i = 0; i < sortFields.length; i++) {
                            var sortField = sortFields[i];

                            /* Obtm o ndice da coluna de acordo com o nome do campo. */
                            var colIndex = cm.findColumnIndex(sortField.field);
                            if (colIndex == -1) {
                                throw "Column not found. Data index = " + field;
                            }

                            /* Obtm o elemento do header e atualiza o cone de sort. */
                            this.changeSortIcon(colIndex, sortField.direction);
                        }
                    },

                    /**
                     * Atualiza o cone de ordenao de uma coluna, possibilitando que multiplos cones de ordenao fiquem visveis ao mesmo tempo.
                     */
                    changeSortIcon : function(colIndex, dir) {
                        var headerElement = Ext.get(this.getHeaderCell(colIndex));

                        /* Se foi definido um dir (ASC ou DESC), adiciona o cone correspondente. */
                        var dirClass;
                        if (dir == 'ASC') {
                            dirClass = this.sortClasses[0];
                        } else if (dir == 'DESC') {
                            dirClass = this.sortClasses[1];
                        } else {
                            return;
                        }

                        headerElement.addClass(dirClass);
                    },

                    /**
                     * Trata os clicks dos menus das colunas
                     * 
                     * @override
                     */
                    handleHdMenuClick : function(item, evt) {
                        var index = this.hdCtxIndex;
                        var order = item.getItemId();

                        switch (order) {
                        case 'asc':
                        case 'desc':
                            this.fireEvent('sortcolumn', index, order, evt);
                            break;
                        default:
                            /* se tiver esconder coluna, basta chamar o super aqui */
                        }
                    },

                    /**
                     * Recalcula e atualiza o scroll da grid.
                     * 
                     * @private
                     * @param scope
                     *            Ext.grid.GridView
                     */
                    refreshScroll : function() {
                        this.calculateVisibleRows();
                        var height = this.calculateScrollerHeight();

                        if (height > 0 && this.scrollWrap != null) {

                            this.scrollWrap.setHeight(height);
                            this.scrollWrap.setStyle({
                                top : this.mainHd.getHeight() + "px"
                            });

                            this.scroll.setHeight(height - this.scrollSizeOffset);
                            
                            this.scrollBt.setHeight(this.scrollSizeOffset);
                            this.scrollbar.setHeight(height * 3);
                            this.adjustScrollerHeight();

                            if (this.getRows()[0]) {
                                this.setRowHeight(this.getRows()[0].offsetHeight);
                                this.verifyNeedShowScroll();
                                this.calculateVisibleRows();
                            } else {
                                this.scrollWrap.setStyle("display", "none");
                            }
                        }
                    },

                    onBeforeRefresh : function(view) {
                        this.keepEditing = this.grid.editing;
                    },

                    /**
                     * Executando quando algum disparar um evento do tipo refresh.
                     * @event refresh
                     */
                    onRefresh : function(view) {

                        /* o keepEditing pode ser true sem ter editor, o Ext deixa nesta situao (this.grid.editing = true) para qdo se vai de uma clula editvel
                         * para uma no, para que se a proxima for editvel o editor voltar */
                        if (this.keepEditing) {
                            if (this.grid.lastActiveEditor) {
                                var recIndex = this.grid.getStore().indexOf(this.grid.lastActiveEditor.record);
                                var curRecIndex = this.grid.getStore().indexOf(this.curRec);
                                var isVisible = recIndex >= curRecIndex && recIndex > -1 && recIndex < (curRecIndex + this.visibleRows);

                                if (isVisible) {
                                    /* o startEditing  feito por defer pois quando vem uma atualizao de dados para a grid,
                                     * rola uma focusCell. Este focusCell  executado pelo EXT em defer, o que acaba fechando o editor */
                                    this.grid.startEditing.defer(1, this.grid, [ recIndex, this.grid.lastActiveEditor.col ]);
                                }
                            } else {
                                /* Se keepEditing for true mas a grid no possuir um editor ativo, significa que a grid estava em edio e agora o foco est em uma clula 
                                 * cujo valor pode ser alterado, mas no possui um editor. Nesse caso o valor da varivel do Ext que informa que a grid est em edio deve 
                                 * ser definido como true para que a grid no saia do modo de edio quando for redesenhada.*/
                                this.grid.editing = true;
                            }
                        }
                    },

                    columnsFixed : function() {
                        /* apenas classe filha utiliza - continua aqui para a grid ser utilizada sem coluna fixa. */
                    },

                    onResize : function(comp, adjWidth, adjHeight, rawWidth, rawHeight) {
                        if (!this.autoHeight && adjHeight) {
                            this.getView().initialHeight = adjHeight;
                        }

                        // Caso o editor esteja ativo quando o usurio redimensionar o navegador ou restaurar/maximizar,
                        // completa o editor para no acontecer de a grid ganhar scroll horizontal e o editor ficar por cima do scroll
                        if (this.activeEditor) {
                            this.activeEditor.completeEdit();
                        }

                        this.view.refreshScroll();
                    },

                    /**
                     * Cria o html dos hints
                     * 
                     * @public
                     * @param target
                     *            HTMLElement ou Ext.Element que receber o hint
                     * @param hints
                     *            lista de hints
                     * @param cellHint
                     *            se for para a clula ento deve ficar ao lado da clula, seno deve seguir o mouse
                     * @return objeto Ext.ToolTip
                     */
                    createHint : function(target, hints, cellHint) {
                        /*
                         * hints[x][y] onde, x = linha do hint no tooltip y = 0 -> icon / y = 1 -> text
                         */
                        var showCellHint = cellHint === undefined ? false : cellHint;/* indica se deve fixar o hint no elemento (celula) ou se deve seguir o mouse */

                        if (hints != null && hints.length > 0) {
                            return TooltipHelper.createHintsHtml(hints, target, !showCellHint);
                        }
                    },

                    /**
                     * @override
                     */
                    ensureVisible : function(row, col, hscroll) {
                        var ret = Ext.ux.senior.grid.GridView.superclass.ensureVisible.call(this, row, col, hscroll);
                        this.onDataNavigate();

                        return ret;
                    },

                    /**
                     * @override
                     */
                    getColumnStyle : function(col, isHeader) {
                        /* Precisa sobrescrever pois todos headers devem ser alinhados  esquerda, independente da configurao */
                        var ret = Ext.ux.senior.grid.GridView.superclass.getColumnStyle.call(this, col, isHeader);
                        if (isHeader) {
                            ret += 'text-align:left;';
                        }
                        return ret;
                    },

                    /**
                     * Verifica se o indice da coluna informada deve exibir checkbox no cabealho.
                     * @param col o indice da coluna
                     * @returns true se  para exibir o checkbox  
                     */
                    hasCheckBox : function(col) {
                        return this.grid.getColumnModel().config[col].showCheckBox;
                    },

                    /**
                     * Verifica se existe alguma coluna fixada na grid.
                     * @returns true se existir alguma coluna fixa
                     */
                    hasFixedColumns : function() {
                        var count = this.grid.getColumnModel().columns.length;
                        for ( var i = 2; i < count; i++) {
                            if (this.grid.getColumnModel().columns[i].fixedColumn) {
                                return true;
                            }
                        }
                        return false;
                    },
                    
                    
                    /**
                     * @override removido if(Ext.isGecko) que no fazia o "defer" ao chamar o focus()
                     */
                    focusCell : function(row, col, hscroll) {
                        this.syncFocusEl(this.ensureVisible(row, col, hscroll));
                        
                        var focusEl = this.focusEl;
                        focusEl.focus.defer(1, focusEl);
                    }

});