/**
 * Responsvel por manipular e controlar o foco do browser.<br>
 * Guarda uma pilha de contextos e sempre atua sobre o topo da pilha.
 * 
 * @author Patrick.Nascimento
 */
FocusManager = {

    /* pilha de contextos. */
    contextStack : [],

    /* ultimo componente focado */
    lastFocused : null,

    /* flag para indicar se os componentes preciso ser reorganizados na lista de tabOrder*/
    pendingReorganization : false,

    /**
     * Inicializa o FocusManager, empilhando um contexto.
     * 
     * @public
     */
    start : function() {
        this.pushContext();
        this.addBodyListener();
    },

    /**
     * Adiciona alguns listeners ao body do html.
     * 
     * @private
     */
    addBodyListener : function() {
        var body = Ext.getBody();

        body.on("keydown", function(event) {
            var key = event.keyCode;

            // ctrl+tab(+shift) deve alternar entre as abas do browser. Safari e Firefox no interceptam o evento, ento temos que ignorar ele aqui.
            if( (Ext.isGecko || Ext.isSafari) && key === Ext.EventObject.TAB && event.ctrlKey){
                return true;            
            }

            /* tab em um lugar desconhecido foca o primeiro componente. */
            if (key === Ext.EventObject.TAB) {
                event.stopEvent();
                this.tabPressed(event);
                return false;
            }

            /* F3 e F4 no devem ir para o browser. */
            if (key === Ext.EventObject.F4 || key === Ext.EventObject.F3) {
                event.stopEvent();
                if (Ext.isIE) {
                    window.event.keyCode = 0;
                }
                return false;
            }
        }, this);
    },

    /**
     * Registra um novo componente no FocusManager.<br>
     * 
     * @param Ext.Component
     *            componente a ser registrado.
     * @param String
     *            grupo de foco no qual este componente pertence.
     * @param int
     *            ordem de foco deste componente.
     * 
     * Chamado pelo ComponenteFocusPlugin para registrar os componentes.
     * 
     * @private
     */
    registerComponent : function(component, focusGroup, focusOrder) {
        var context = this.getContext();

        /* se no existe este grupo ainda, cria. */
        var group = context.getGroup(focusGroup);
        if (group === null) {
            group = context.addGroup(focusGroup);
        }
        if (!group.containsComponent(component)) {

            /* SLogger.warn("[registerComponent] ", component.toString()); */
            group.add(component, focusOrder);

            /* registra o listener para a tecla Tab. */
            this.registerListener(this, component);
        }
    },

    /**
     * Procura um prximo componente editvel depois deste painel para setar o foco.<br>
     * Atualiza a lista de filhos do painel (panel.allChildren), avisando que estes componentes no podem receber foco.
     * 
     * Chamado pelo ContainerFocusPlugin quando um painel  fechado.
     * 
     * @private
     */
    collapsePanel : function(panel) {
        /* busca o ltimo componente da lista para achar o prximo focavel. */
        var lastComponent;
        if (panel.allFocusableChildren) {
            lastComponent = panel.allFocusableChildren[panel.allFocusableChildren.length - 1];
        } else {
            lastComponent = -1;
        }

        var context = this.getContext();
        var node = context.getNodeToFocus(lastComponent);

        /* ainda no renderizou todos os componentes. */
        if (node === null) {
            return;
        }
        
        /* se ningum estava focado antes, continua no focado */
        if (!FocusManager.getFocusedComponent()) {
            return;
        }

        /* a partir do ltimo componente daquele panel busca o prximo focavel. */
        while (!this.canFocusComponent(node.getComponent())) {
            node = node.getNext();

            if (node == null) {
                /* foca no primeiro da lista circular (defaultGroup) */
                this.focusFirstComponent();
                return;
            }

            if (node.getComponent() === lastComponent) {
                /* deu a volta e no achou ningum, ento nada faz. */
                context.getActiveGroup().setCurrentNode(null);
                context.setActiveGroup(null);
                return;
            }
        }

        /* atualiza o n corrente. */
        context.getActiveGroup().setCurrentNode(node);

        /* seta o foco no componente */
        this.focusComponent(node.getComponent());
    },

    /**
     * Registra o listener de tecla Tab para o componente.
     * 
     * @private
     */
    registerListener : function(scope, component) {
        var keyNav = new Ext.KeyNav(component.getEl(), {
            "tab" : function(e) {
                // no safari no troca de abas, fica navegando dentro da
                // tela
                if ((Ext.isSafari || Ext.isGecko) && e.ctrlKey) {
                    return true;
                }
                Ext.EventObject.stopEvent();
                component.fireEvent('specialkey', component, e);
                return this.tabPressed(e);
            },
            scope : scope
        });
        /* guarda no componente para depois desregistrar o keynav. */
        component.keyNav = keyNav;
    },

    /**
     * Listener para evento de tecla Tab pressionada.
     * 
     * @private
     */
    tabPressed : function(event) {
        /* tab ou shift+tab */

        if (this.pendingReorganization) {
            this.rearrangeTabOrder();
            this.pendingReorganization = false;
        }

        var shiftKey = event.shiftKey;

        var context = this.getContext();
        var group = context.getActiveGroup();
        var node = group.getCurrentNode();

        if (group.getCurrentNode() != null) {
            if (!ScreenManager.isInterfaceBlocked()) {
                this.lastFocused = group.getCurrentNode().getComponent();
                var component = node.getComponent();
                if (component.selectText && this.hasDOMFocus(component)) {
                    /* limpa seleo do edit antigo. */
                    component.selectText(0, 0);
                }

                /*SLogger.error("[tabPressed] (before) ", component.toString());*/

                do {
                    /* busca o prximo n da navegao */
                    node = shiftKey ? node.getPrev() : node.getNext();

                    /* lista simples e no achou prximo. */
                    if (node == null) {
                        /* foca no primeiro da lista circular (defaultGroup) */
                        this.focusFirstComponent();
                        return false;
                    }

                    if (node.getComponent() === this.getFocusedComponent()) {
                        break;
                    }

                } while (!(!this.isFocusedComponent(node.getComponent()) && this.canFocusComponent(node.getComponent())));

                /*SLogger.error("[tabPressed] (after) ", node.component.toString());*/
                /* achou algum, foca nele. */
                this.focusNode(group, node, event);
            }
            /* cancela evento. */
            return false;
        } else {
            this.focusFirstComponent();
        }
    },

    /**
     * Verifica se o componente est focado no documento
     * @param component
     * @returns.
     */
    hasDOMFocus : function(component) {
        if (!document.activeElement) { // no IE fica 'null' em algumas situaes
            return false;
        }
        return document.activeElement.id == component.getId();
    },

    /**
     * Seta o foco em um Node e faz as devidas atualizaes nas listas.
     * 
     * @private
     */
    focusNode : function(group, node, event) {
        this.focusComponent(node.getComponent(), event);
        group.setCurrentNode(node);
    },

    /**
     * Seta o foco em um determinado componente. <br>
     * Caso seja um Ext.Field, tambm seleciona o texto. *
     * 
     * @param Ext.Component
     *            componente a receber o foco.
     * 
     * @private
     */
    focusComponent : function(component, event) {
        /* SLogger.fatal("[focusComponent] ", component.toString()); */
        //a dupla negao garante que a varivel selecText receber um valor booleano
        var selectText = !!component.el.dom.select;
        SLogger.debug("Component : " + component.xtype);
        // se for um TextField (ou outro equivalente) o usurio perder a digitao quando selecionar o texto, ento verifica-se se o mesmo j no est com foco
        selectText = selectText && !this.hasDOMFocus(component);
        component.focus(selectText, null, event);
    },

    /**
     * Chamado quando um componente recebe o foco. <br>
     * 
     * Um componente pode receber foco atravs do FocusManager ou por um clique neste componente.
     * 
     * @param Ext.Component
     *            componente que recebeu o foco.
     * 
     * @public
     */
    onComponentFocus : function(component) {
        var context = this.getContext();
        var group = context.getActiveGroup();

        if (!group) {
            return; /* no tem grupo. */
        }

        if (group.getCurrentNode() != null) {
            this.lastFocused = group.getCurrentNode().getComponent();
        }

        var curNode = group.getCurrentNode();

        if (curNode && component === curNode.getComponent()) {
            /* evento que partiu do focusmanager - ignora. */
        } else {
            /* SLogger.info("[onComponentFocus] ", component.toString()); */
            /* evento que veio do clike em algum component - j est focado */
            var node = context.getNodeToFocus(component);

            /* pode ter mudado o grupo. */
            group = context.getActiveGroup();
            group.setCurrentNode(node);
        }
    },

    /**
     * Reinicia o controle de foco.
     * 
     * @public
     */
    reset : function() {
        var context = this.getContext();
        var group = context.getActiveGroup();
        group.setCurrentNode(null);

    },

    /**
     * Retorna qual componente est focado logicamente no FocusManager.
     * 
     * @public
     */
    getFocusedComponent : function() {
        var context = this.getContext();
        var group = context.getActiveGroup();
        if (group !== null && group.getCurrentNode() !== null) {
            return group.getCurrentNode().getComponent();
        }
        return null;
    },

    /**
     * Remove um componente do FocusManager.
     * 
     * @param Ext.Component
     *            componente a ser removido.
     * 
     * @public
     */
    removeComponent : function(component) {
        var context = this.getContext();
        context.removeComponent(component);
    },

    /**
     * Define o foco ao primeiro componente focavel do 'defaultGroup' deste contexto.<br>
     * Caso no encontre um componente focavel apenas retorna.
     * 
     * @public
     */
    focusFirstComponent : function() {
        var context = this.getContext();

        /* busca o grupo atual. */
        var group = context.getActiveGroup();
        if (!group) {
            return;
        }
        if (group != context.getDefaultGroup()) {
            /* limpa o component corrente. */
            group.setCurrentNode(null);
        }

        group = context.getDefaultGroup();

        context.setActiveGroup(group);

        var first = group.getFirst();
        var node = first;

        /* busca o primeiro editvel. */
        while (node != null && !this.canFocusComponent(node.getComponent())) {
            node = node.getNext();

            /* no encontrou ningum editvel */
            if (node == first) {
                return;
            }
        }

        if (node == null) {
            return;
        }

        /* SLogger.debug("[focusFirstComponent] ", node.component.toString()); */

        /* seta o foco no primeiro */
        this.focusNode(group, node);
    },

    /**
     * Retorna o foco ao componente corrente deste contexto. <br>
     * Utilizado para garantir que o componente certo est focado.
     * 
     *  chamado logo aps um modal ser fechado e a cada vez que uma requisio volta do servidor.
     * 
     * @public
     */
    focusCurrentComponent : function() {
        var context = this.getContext();
        var group = context.getActiveGroup();

        if (!group) {
            return; /* no tem grupo. */
        }

        var node = group.getCurrentNode();
        if (node && node.getComponent() && node.getComponent().getFocusOrder != -1) {
            /*SLogger.debug("[focusCurrentComponent] ", node.component.toString());*/
            /* se existia um corrente, foca nele. */
            this.focusNode(group, node);
        } else {
            /* seno foca no primeiro mesmo. */
            this.focusFirstComponent();
        }

    },

    /**
     * Foca o prximo componente deste contexto.
     * 
     * @public
     */
    focusNextComponent : function(e) {
        /* pega o objeto de evento corrente */
        e = e || Ext.EventObject;
        /* SLogger.debug("[focusNextComponent] "); */
        this.tabPressed(e);
    },

    /**
     * Foca o componente anterior deste contexto.
     * 
     * @public
     */
    focusPrevComponent : function(e) {
        /* pega o objeto de evento corrente */
        e = e || Ext.EventObject;
        /* SLogger.debug("[focusPrevComponent] "); */
        /* para focar o anterior, basta dizer que o shift est pressionado */
        e.shiftKey = true;
        this.tabPressed(e);
    },

    /**
     * Verifica se pode setar foco em um componente.
     * 
     * @private
     */
    canFocusComponent : function(component) {
        return component.focableComponent && component.isVisible() && !component.disabled && !component.isPanelDeactivate() && component.getFocusOrder() > 0;
    },

    /**
     * Verifica se o componente possui foco
     * @param component componente que ser verificado
     */
    isFocusedComponent : function(component) {
        return this.getFocusedComponent() === component;
    },

    /**
     * Empilha um novo ComponentContext.
     * 
     * @public
     */
    pushContext : function() {
        var context = new FocusContext();
        this.contextStack.push(context);
    },

    /**
     * Remove novo ComponentContext da pilha.
     * 
     * @public
     */
    popContext : function() {
        var context = this.contextStack.pop();
        context.release();
    },

    /**
     * Retorna o contexto corrente.
     * 
     * @private
     */
    getContext : function() {
        return this.contextStack[this.contextStack.length - 1];
    },

    /**
     * Libera os recursos do FocusManager.
     * 
     * @public
     */
    release : function() {
        this.popContext();
    },

    getLastFocused : function() {
        return this.lastFocused;
    },

    /**
     * Rearranja a lista de componentes de acordo com o TabOrder
     * 
     * @public
     */
    rearrangeTabOrder : function() {
        FocusManager.getContext().getDefaultGroup().sortByOrder();
    },

    /**
     * Configura a flag para reorganizao do tabOrder
     */
    setPendingReorganization : function(pendingOrganization) {
        this.pendingReorganization = pendingOrganization;
    },

    containsComponent : function(component) {
        var allGroups = FocusManager.getContext().allGroups;
        for ( var id in allGroups) {
            var group = allGroups[id];
            if (group.containsComponent && group.containsComponent(component)) {
                return true;
            }
        }
        return false;
    }

};
FocusManager.start();