function FocusContext() {
    this.allGroups = {};
    this.activeGroup = null;
}

/**
 * Um contexto  formado por vrios grupos de foco, onde cada grupo  uma lista de ns ordenados de acordo com o focusOrder dos componentes destes ns.<br>
 * Um contexto representa uma perspectiva/viso ou uma tela modal. O contexto tambm tem um grupo padro, que  uma lista circular.
 * 
 * O contexto tambm  responsvel por manter qual grupo de foco  o ativo no momento, para que o FocusManager saiba em quais componentes navegar.
 * 
 * @author Patrick.Nascimento
 */
FocusContext.prototype = {

    /* array com os grupos deste contexto. */
    allGroups : {},

    /* grupo ativo. */
    activeGroup : null,

    /* id do grupo padro. */
    idDefaultGroup : "defaultGroup",

    construtor : FocusContext,

    /**
     * Seta o grupo ativo deste contexto.
     * 
     * @param Object
     *            o grupo ativo.
     */
    setActiveGroup : function(group) {
        this.activeGroup = group;
    },

    /**
     * Retorna o grupo ativo deste contexto.<br>
     * Caso no existir, retorna o grupo padro.
     * 
     * @return Object o grupo ativo.
     */
    getActiveGroup : function() {
        if (!this.activeGroup) {
            return this.getDefaultGroup();
        }
        return this.activeGroup;
    },

    /**
     * Retorna o grupo padro deste contexto.
     * 
     * @return o grupo.
     */
    getDefaultGroup : function() {
        var group = this.getGroup(this.idDefaultGroup);
        /* se o grupo padro no existe, ele  criado */
        if(group == null) {
        	group = this.addGroup(this.idDefaultGroup);
        }
        return group;
    },

    /**
     * Retorna um grupo de acordo com id.
     * 
     * @param String
     *            o id do grupo
     * @return o grupo.
     */
    getGroup : function(id) {
        if (this.allGroups[id]) {
            return this.allGroups[id];
        }
        return null;
    },

    /**
     * Cria e adiciona um novo grupo ao contexto.
     * 
     * @param String
     *            o id do grupo
     * @return Object o grupo que foi criado.
     */
    addGroup : function(id) {
        var group;
        if (id == this.idDefaultGroup) {
            group = new CircleList(id);
        } else {
            group = new SimpleList(id);
        }

        this.allGroups[id] = group;
        return group;
    },

    /**
     * Procura um n a partir de um componente e define o grupo deste n como o grupo corrente deste contexto.
     * 
     * @param Ext.Component
     *            componente a ser buscado o n referente.
     * @return Node n referente a este componente.
     */
    getNodeToFocus : function(component) {

        var group = null, toReturn = null;
        for ( var id in this.allGroups) {
            group = this.allGroups[id];

            var node = group.getNode(component);
            if (node == null) {
                /* garante que no h ningum focado neste grupo. */
                group.setCurrentNode(null);
            } else {
                toReturn = node;
                this.activeGroup = group;
            }
        }

        return toReturn;
    },

    /**
     * Remove um componente do controle de foco.<br>
     *  percorrido todos os grupos e verificado se o componente pertence  algum. Caso pertena,  removido.
     * 
     * @param Ext.Component
     *            componente a ser removido.
     */
    removeComponent : function(component) {
        var group = null;
        for ( var id in this.allGroups) {
            group = this.allGroups[id];

            var node = group.getNode(component);
            if (node !== null) {
                group.removeNode(node);
                return;
            }
        }

    },

    /**
     * Libera recursos do FocusContext.
     */
    release : function() {
        var group = null;
        for ( var id in this.allGroups) {
            group = this.allGroups[id];

            if (group.release) {
                group.release();
            }
            delete group[id];
        }

        delete this.allGroups;
        delete this.activeGroup;
    }

};
