ColorUtils = {
/**
	 * Convert HSV color format to RGB color format
	 * @param {Integer/Array(h, s, v)} h
	 * @param {Integer} s (optional)
	 * @param {Integer} v (optional)
	 * @return {Array}
	 */
	hsvToRgb: function(h, s, v) {
		if(h instanceof Array) { return this.hsvToRgb.call(this, h[0], h[1], h[2]); }
		var r, g, b, i, f, p, q, t;
	    i = Math.floor((h / 60) % 6);
	    f = (h / 60) - i;
	    p = v * (1 - s);
	    q = v * (1 - f * s);
	    t = v * (1 - (1 - f) * s);
	    switch(i) {
	        case 0: r=v; g=t; b=p; break;
	        case 1: r=q; g=v; b=p; break;
	        case 2: r=p; g=v; b=t; break;
	        case 3: r=p; g=q; b=v; break;
	        case 4: r=t; g=p; b=v; break;
	        case 5: r=v; g=p; b=q; break;
	    }
	    return [this.realToDec(r), this.realToDec(g), this.realToDec(b)];
	},
	/**
	 * Convert RGB color format to HSV color format
	 * @param {Integer/Array(r, g, b)} r
	 * @param {Integer} g (optional)
	 * @param {Integer} b (optional)
	 * @return {Array}
	 */
	rgbToHsv: function(r, g, b) {
		if(r instanceof Array) { return this.rgbToHsv.call(this, r[0], r[1], r[2]); }
        r = r / 255;
        g = g / 255;
        b = b / 255;
        var min, max, delta, h, s, v;
        min = Math.min(Math.min(r, g), b);
        max = Math.max(Math.max(r, g), b);
        delta = max - min;
        switch (max) {
            case min: h = 0; break;
            case r:   h = 60 * (g - b) / delta;
                      if (g < b) { h += 360; }
                      break;
            case g:   h = (60 * (b - r) / delta) + 120; break;
            case b:   h = (60 * (r - g) / delta) + 240; break;
        }
        s = (max === 0) ? 0 : 1 - (min / max);
        return [Math.round(h), s, max];
	},
	/**
	 * Convert a float to decimal
	 * @param {Float} n
	 * @return {Integer}
	 */
	realToDec: function(n) {
		return Math.min(255, Math.round(n * 255));
	},
	/**
	 * Convert RGB color format to Hexa color format
	 * @param {Integer/Array(r, g, b)} r
	 * @param {Integer} g (optional)
	 * @param {Integer} b (optional)
	 * @return {String}
	 */
	rgbToHex: function(r, g, b) {
		if(r instanceof Array) { return this.rgbToHex.call(this, r[0], r[1], r[2]); }
		return '#'+ this.decToHex(r) + this.decToHex(g) + this.decToHex(b);
	},
	/**
	 * Convert an integer to hexa
	 * @param {Integer} n
	 * @return {String}
	 */
	decToHex: function(n) {
		var HCHARS = '0123456789ABCDEF';
        n = parseInt(n, 10);
        n = (!isNaN(n)) ? n : 0;
        n = (n > 255 || n < 0) ? 0 : n;
        return HCHARS.charAt((n - n % 16) / 16) + HCHARS.charAt(n % 16);
	},
	/**
	 * Return with position of a character in this.HCHARS string
	 * @private
	 * @param {Char} c
	 * @return {Integer}
	 */
	getHCharPos: function(c) {
		var HCHARS = '0123456789ABCDEF';
		return HCHARS.indexOf(c.toUpperCase());
	},
	/**
	 * Convert a hexa string to decimal
	 * @param {String} hex
	 * @return {Integer}
	 */
	hexToDec: function(hex) {
        var s = hex.split('');
        return ((this.getHCharPos(s[0]) * 16) + this.getHCharPos(s[1]));
	},
	/**
	 * Convert a hexa string to RGB color format
	 * @param {String} hex
	 * @return {Array}
	 */
	hexToRgb: function(hex) {
		return [ this.hexToDec(hex.substr(0, 2)), this.hexToDec(hex.substr(2, 2)), this.hexToDec(hex.substr(4, 2)) ];
	},
	/**
	 * Convert Y coordinate to HUE value
	 * @private
	 * @param {Integer} y
	 * @return {Integer}
	 */
	getHue: function(y) {
		var hue = 360 - Math.round(((181 - y) / 181) * 360);
		return hue === 360 ? 0 : hue;
	},
	/**
	 * Convert HUE value to Y coordinate
	 * @private
	 * @param {Integer} hue
	 * @return {Integer}
	 */
	getHPos: function(hue) {
		return 181 - hue * (181 / 360);
	},
	/**
	 * Convert X coordinate to Saturation value
	 * @private
	 * @param {Integer} x
	 * @return {Integer}
	 */
	getSaturation: function(x) {
		return x / 181;
	},
	/**
	 * Convert Saturation value to Y coordinate
	 * @private
	 * @param {Integer} saturation
	 * @return {Integer}
	 */
	getSPos: function(saturation) {
		return saturation * 181;
	},
	/**
	 * Convert Y coordinate to Brightness value
	 * @private
	 * @param {Integer} y
	 * @return {Integer}
	 */
	getValue: function(y) {
		return (181 - y) / 181;
	},
	/**
	 * Convert Brightness value to Y coordinate
	 * @private
	 * @param {Integer} value
	 * @return {Integer}
	 */
	getVPos: function(value) {
		return 181 - (value * 181);
	},
	
	/**
	 * Not documented yet
	 */
	invert: function(r, g, b) {
		if(r instanceof Array) { return this.invert.call(this, r[0], r[1], r[2]); }
		return [255-r,255-g,255-b];
	},
	
	/**
	 * Not documented yet
	 */
	checkSafeNumber: function(v) {
	    if (!isNaN(v)) {
	        v = Math.min(Math.max(0, v), 255);
	        var i, next;
	        for(i=0; i<256; i=i+51) {
	            next = i + 51;
	            if (v>=i && v<=next) { return (v - i > 25) ? next : i; }
	        }
	    }
	    return v;
	},
	/**
	 * Not documented yet
	 */
	websafe: function(r, g, b) {
		if(r instanceof Array) { return this.websafe.call(this, r[0], r[1], r[2]); }
		return [this.checkSafeNumber(r), this.checkSafeNumber(g), this.checkSafeNumber(b)];
	}
}