/**
 * @class Ext.ux.ColorPicker
 * @extends Ext.BoxComponent
 * This is a color picker.
 * @author: Amon
 * @constructor
 * Creates a new ColorPicker
 * @param {Object} config Configuration options
 * @version 1.1.2
 * 
 * license: LGPLv3
 */

Ext.namespace('Ext.ux');
/**
 *
 */
Ext.ux.ColorPicker = Ext.extend(Ext.BoxComponent, {
	/**
	 *
	 */
	initComponent: function() {
		this.applyDefaultsCP();
		Ext.ux.ColorPicker.superclass.initComponent.apply(this, arguments);
		this.addEvents('select');
	},
	
	/**
	 *
	 */
	onRender: function() {
		Ext.ux.ColorPicker.superclass.onRender.apply(this, arguments);
		// check if container, self-container or renderTo exists
		this.body = this.body || (this.container || (this.renderTo || Ext.DomHelper.append(Ext.getBody(), {}, true)));
		if(!this.el) {
			this.el = this.body;
			if(this.cls) { Ext.get(this.el).addClass(this.cls); }
		}
		// render this component
		// create RGB Slider
		var rgbEl = Ext.DomHelper.append(this.body, {
			id: this.getCmpId('rgb'),
			cls: 'x-cp-rgbpicker'
		});
		this.rgb = Ext.get(rgbEl);
		// Create HUE Slider
		var hueEl = Ext.DomHelper.append(this.body, {
			id: this.getCmpId('hue'),
			cls: 'x-cp-huepicker'
		});
		this.hue = Ext.get(hueEl);
		// Initialize HUE Picker DD
		this.huePicker = Ext.DomHelper.append(this.body, { 'cls': 'x-cp-hueslider' });
		this.hueDD = new Ext.dd.DD(this.huePicker, 'huePicker');
		this.hueDD.constrainTo(this.hue, {'top':-7,'right':-3,'bottom':-7,'left':-3});
		this.hueDD.onDrag = this.moveHuePicker.createDelegate(this);
		// initialize onclick on the rgb picker
		this.hue.on('mousedown', this.clickHUEPicker.createDelegate(this));
		// initialize start position
		Ext.get(this.huePicker).moveTo(this.hue.getLeft() - 3, this.hue.getTop() - 7);
		// Initialize RGB Picker DD
		this.rgbPicker = Ext.DomHelper.append(this.body, { 'cls': 'x-cp-rgbslider' });
		this.rgbDD = new Ext.dd.DD(this.rgbPicker, 'rgbPicker');
		this.rgbDD.constrainTo(this.rgb, -7);
		this.rgbDD.onDrag = this.moveRGBPicker.createDelegate(this);
		// initialize onclick on the rgb picker
		this.rgb.on('mousedown', this.clickRGBPicker.createDelegate(this));
		// initialize start position
		Ext.get(this.rgbPicker).moveTo(this.rgb.getLeft() - 7, this.rgb.getTop() - 7);
		// Create color divs and Form elements
		Ext.DomHelper.append(this.body, {
							id: this.getCmpId('fCont'),
							cls: 'x-cp-formcontainer'
						}, true);
		
		new Ext.form.FormPanel({
			renderTo: this.getCmpId('fCont'),
			frame: true,
			labelAlign: 'left',
			labelWidth: 10,
			forceLayout: true,
			items: [{
				layout: 'column',
				items: [{
					columnWidth: .5,
					layout: 'form',
					defaultType: 'numberfield',
					defaults: {
						width: 30,
						value: 0,
						minValue: 0,
						maxValue: 255,
						allowBlank: false,
						labelSeparator: ''
					},
					items: [{
						fieldLabel: 'R',
						id: this.getCmpId('iRed')
					},{
						fieldLabel: 'G',
						id: this.getCmpId('iGreen')
					},{
						fieldLabel: 'B',
						id: this.getCmpId('iBlue')
					}]
				},{
					columnWidth: .5,
					layout: 'form',
					defaultType: 'numberfield',
					defaults: {
						width: 30,
						value: 0,
						minValue: 0,
						maxValue: 255,
						allowBlank: false,
						labelSeparator: ''
					},
					items: [{
						fieldLabel: 'H',
						maxValue: 360,
						id: this.getCmpId('iHue')
					},{
						fieldLabel: 'S',
						id: this.getCmpId('iSat')
					},{
						fieldLabel: 'V',
						id: this.getCmpId('iVal')
					}]
				}]
			}]
		});
		Ext.getCmp(this.getCmpId('iRed')).on('change', this.updateFromIRGB, this);
		Ext.getCmp(this.getCmpId('iGreen')).on('change', this.updateFromIRGB, this);
		Ext.getCmp(this.getCmpId('iBlue')).on('change', this.updateFromIRGB, this);
		Ext.getCmp(this.getCmpId('iHue')).on('change', this.updateFromIHSV, this);
		Ext.getCmp(this.getCmpId('iSat')).on('change', this.updateFromIHSV, this);
		Ext.getCmp(this.getCmpId('iVal')).on('change', this.updateFromIHSV, this);
		
	    Ext.DomHelper.append(this.getCmpId('fCont'), { 'cls': 'x-cp-colorbox', 'id': this.getCmpId('cWebSafe') }, true).update('Websafe');
		Ext.DomHelper.append(this.getCmpId('fCont'), { 'cls': 'x-cp-colorbox', 'id': this.getCmpId('cInverse') }, true).update('Inverter');
		Ext.DomHelper.append(this.getCmpId('fCont'), { 'cls': 'x-cp-colorbox', 'id': this.getCmpId('cColor') }, true).update('Escolher Cor');
		
		Ext.get(this.getCmpId('cWebSafe')).on('click', this.updateFromBox, this);
		Ext.get(this.getCmpId('cInverse')).on('click', this.updateFromBox, this);
		Ext.get(this.getCmpId('cColor')).on('click', this.selectColor, this);
		Ext.DomHelper.append(this.body, {'tag':'br','cls':'x-cp-clearfloat'});
	},
	/**
	 *
	 */
	applyDefaultsCP: function() {
		Ext.apply(this, {
			cls: 'x-cp-mainpanel',
			resizable: this.resizable || false,
			HSV: {
				h: 0,
				s: 0,
				v: 0
			},
			updateMode: null
		});
	},
	
	/**
	 *
	 */
	updateRGBPosition: function(x, y) {
		this.updateMode = 'click';
		x = x < 0 ? 0 : x;
		x = x > 181 ? 181 : x;
		y = y < 0 ? 0 : y;
		y = y > 181 ? 181 : y;
		this.HSV.s = ColorUtils.getSaturation(x);
		this.HSV.v = ColorUtils.getValue(y);
		Ext.get(this.rgbPicker).moveTo(this.rgb.getLeft() + x - 7, this.rgb.getTop() + y - 7, (this.animateMove || true));
		this.updateColor();
	},
	/**
	 *
	 */
	updateHUEPosition: function(y) {
		this.updateMode = 'click';
		y = y < 1 ? 1 : y;
		y = y > 181 ? 181 : y;
		this.HSV.h = Math.round(360 / 181 * (181 - y));
		Ext.get(this.huePicker).moveTo(Ext.get(this.huePicker).getLeft(), this.hue.getTop() + y - 7, (this.animateMove || true));
		this.updateRGBPicker(this.HSV.h);
		this.updateColor();
	},
	/**
	 *
	 */
	clickRGBPicker: function(event, element) {
		this.updateRGBPosition(event.xy[0] - this.rgb.getLeft() , event.xy[1] - this.rgb.getTop());
	},
	/**
	 *
	 */
	clickHUEPicker: function(event, element) {
		this.updateHUEPosition(event.xy[1] - this.hue.getTop());
	},
	/**
	 *
	 */
	moveRGBPicker: function(event) {
		this.rgbDD.constrainTo(this.getCmpId('rgb'), -7);
		this.updateRGBPosition(Ext.get(this.rgbPicker).getLeft() - this.rgb.getLeft() + 7 , Ext.get(this.rgbPicker).getTop() - this.rgb.getTop() + 7);
	},
	/**
	 *
	 */
	moveHuePicker: function(event) {
		this.hueDD.constrainTo(this.getCmpId('hue'), {'top':-7,'right':-3,'bottom':-7,'left':-3});
		this.updateHUEPosition(Ext.get(this.huePicker).getTop() - this.hue.getTop() + 7);
	},
	/**
	 *
	 */
	updateRGBPicker: function(newValue) {
		this.updateMode = 'click';
		this.rgb.setStyle({ 'background-color': ColorUtils.rgbToHex(ColorUtils.hsvToRgb(newValue, 1, 1)) });
		this.updateColor();
	},
	/**
	 *
	 */
	updateColor: function() {
		var rgb = ColorUtils.hsvToRgb(this.HSV.h, this.HSV.s, this.HSV.v);
		var websafe = ColorUtils.websafe(rgb);
		var invert = ColorUtils.invert(rgb);
		var wsInvert = ColorUtils.invert(websafe);
		if(this.updateMode !== 'hexa') {
			this.hexaColor = ColorUtils.rgbToHex(rgb);
			this.fireEvent('colorchanged', this.hexaColor);
		}
		if(this.updateMode !== 'rgb') {
			Ext.getCmp(this.getCmpId('iRed')).setValue(rgb[0]);
			Ext.getCmp(this.getCmpId('iGreen')).setValue(rgb[1]);
			Ext.getCmp(this.getCmpId('iBlue')).setValue(rgb[2]);
		}
		if(this.updateMode !== 'hsv') {
			Ext.getCmp(this.getCmpId('iHue')).setValue(Math.round(this.HSV.h));
			Ext.getCmp(this.getCmpId('iSat')).setValue(Math.round(this.HSV.s * 100));
			Ext.getCmp(this.getCmpId('iVal')).setValue(Math.round(this.HSV.v * 100));
		}
		Ext.get(this.getCmpId('cColor')).setStyle({
			'background': ColorUtils.rgbToHex(rgb),
			'color': ColorUtils.rgbToHex(invert)
		});
		Ext.getDom(this.getCmpId('cColor')).title = ColorUtils.rgbToHex(rgb);
		Ext.get(this.getCmpId('cInverse')).setStyle({
			'background': ColorUtils.rgbToHex(invert),
			'color': ColorUtils.rgbToHex(rgb)
		});
		Ext.getDom(this.getCmpId('cInverse')).title = ColorUtils.rgbToHex(invert);
		Ext.get(this.getCmpId('cWebSafe')).setStyle({
			'background': ColorUtils.rgbToHex(websafe),
			'color': ColorUtils.rgbToHex(wsInvert)
		});
		Ext.getDom(this.getCmpId('cWebSafe')).title = ColorUtils.rgbToHex(websafe);
		Ext.get(this.huePicker).moveTo(Ext.get(this.huePicker).getLeft(), this.hue.getTop() + ColorUtils.getHPos(Ext.getCmp(this.getCmpId('iHue')).getValue()) - 7, (this.animateMove || true));
		Ext.get(this.rgbPicker).moveTo(this.rgb.getLeft() + ColorUtils.getSPos(Ext.getCmp(this.getCmpId('iSat')).getValue() / 100) - 7, this.hue.getTop() + ColorUtils.getVPos(Ext.getCmp(this.getCmpId('iVal')).getValue() / 100) - 7, (this.animateMove || true));
		this.rgb.setStyle({ 'background-color': ColorUtils.rgbToHex(ColorUtils.hsvToRgb(Ext.getCmp(this.getCmpId('iHue')).getValue(), 1, 1)) });
	},
	/**
	 *
	 */
	setColor: function(c) {
		c = c.replace('#', '');
		if(!/^[0-9a-fA-F]{6}$/.test(c))return;
		this.hexaColor = c;
		this.updateFromIHexa();
	},
	/**
	 *
	 */
	updateFromIRGB: function(input, newValue, oldValue) {
		this.updateMode = 'rgb';
		var temp = ColorUtils.rgbToHsv(Ext.getCmp(this.getCmpId('iRed')).getValue(), Ext.getCmp(this.getCmpId('iGreen')).getValue(), Ext.getCmp(this.getCmpId('iBlue')).getValue());
		this.HSV = { h: temp[0], s:temp[1], v:temp[2]};
		this.updateColor();
	},
	
	/**
	 *
	 */
	updateFromIHSV: function(input, newValue, oldValue) {
		this.updateMode = 'hsv';
		this.HSV = {
				h: Ext.getCmp(this.getCmpId('iHue')).getValue(),
				s: Ext.getCmp(this.getCmpId('iSat')).getValue() / 100,
				v: Ext.getCmp(this.getCmpId('iVal')).getValue() / 10
		};
		this.updateColor();
	},
	
	/**
	 *
	 */
	updateFromIHexa: function(input, newValue, oldValue) {
		this.updateMode = 'hexa';
		var temp = ColorUtils.rgbToHsv(ColorUtils.hexToRgb(this.hexaColor));
		this.HSV = { h: temp[0], s:temp[1], v:temp[2]};
		this.updateColor();
	},
	
	/**
	 *
	 */
	updateFromBox: function(event, element) {
		this.updateMode = 'click';
		var col = Ext.get(element).getColor('backgroundColor', '', '');
		col = col.replace("#", "");
		var temp = ColorUtils.rgbToHsv(ColorUtils.hexToRgb(col));
		this.HSV = { h: temp[0], s:temp[1], v:temp[2]};
		this.updateColor();
	},

	selectColor: function(event, element) {
		this.fireEvent('select', this, Ext.get(element).getColor('backgroundColor', '', ''));
	},
	
	getCmpId: function(postfix) {
		return this.getId() + '_' + ( postfix || 'cp' );
	}
});

Ext.reg('seniorcolorpicker', Ext.ux.ColorPicker);