Ext.namespace('Ext.ux.senior.form');

/**
 * Trigger Field para edio de campos do Tipo CLOB
 * 
 * @author Joao.Maas
 */
Ext.ux.senior.form.TriggerTextAreaField = Ext.extend(Ext.form.TriggerField, {

    map : null,
    editText : '',
    // no utilizamos validao do Ext, passando false otimiza por no gerar chamadas na classe base    
    validationEvent : false,
    showingModal : false,
    fieldClass : "x-form-field no-clear-button",

    /**
     * @Override
     */
    initComponent : function() {
        Ext.ux.senior.form.TriggerTextAreaField.superclass.initComponent.call(this, arguments);
        this.onTriggerClick = this.openMemo;
        this.enableKeyEvents = true;
        this.triggerClass = 'x-senior-memo-trigger';
    },

    /**
     * @Override
     */
    onRender : function(ct, position) {
        Ext.ux.senior.form.TriggerTextAreaField.superclass.onRender.call(this, ct, position);
        this.getEl().on("keydown", this.onEdit, this);
		if (this.gridEditor) {
        	this.gridEditor.completeOnEnter = false;
        }
    },
    
    /**
     * @plugin EventManagerPlugin
     */
    eventMapping : function() {
        return {
            valuechanged : this.valueChanged
        };
    },

    /**
     * @event valuechanged
     */
    valueChanged : function(value) {
        return {
            type : 'changed',
            value : value
        };
    },

    /**
     * @event Ext.ux.senior.form.TriggerTextAreaField.EditorWindow#okClick
     */
    changeValue : function(value) {
        this.fireEvent('valuechanged', value);
    },

    /**
     * @event keypress, onTriggerClick
     */
    onEdit : function(evt) {
        /*
         * Faz tratamento das teclas que no devem abrir o modo de edio do campo. Este tratamento  feito pois, para o valor digitado pelo usurio
         * no EditBox passar para a TextArea  necessrio que o evento de keyDown seja interceptado. E como no keyDown todas as teclas do teclado so
         * consideradas, fica mais fcil selecionar as execees.
         * 
         */
        switch (evt.getKey()) {
        case Ext.EventObject.F1:
        case Ext.EventObject.F2:
        case Ext.EventObject.F3:
        case Ext.EventObject.F5:
        case Ext.EventObject.F6:
        case Ext.EventObject.F7:
        case Ext.EventObject.F8:
        case Ext.EventObject.F9:
        case Ext.EventObject.F10:
        case Ext.EventObject.F11:
        case Ext.EventObject.F12:
        case Ext.EventObject.CTRL:
        case Ext.EventObject.TAB:
        case Ext.EventObject.SHIFT:
        case Ext.EventObject.ESC:
        case Ext.EventObject.ALT:
        case Ext.EventObject.RIGHT:
        case Ext.EventObject.LEFT:
        case Ext.EventObject.DOWN:
        case Ext.EventObject.UP:
        case Ext.EventObject.HOME:
        case Ext.EventObject.END:
            return;
        }
        this.openMemo();
    },

    /**
     * @private
     */
    openMemo : function() {
        Ext.ux.senior.form.TriggerTextAreaField.EditorWindow.open(this);
        this.showingModal = true;
    },

    validateBlur : function(e) {
        return !this.showingModal;
    },

    /**
     * @Override
     */
    setValue : function(value) {
	    this.originalValue = value;
        var displayValue = this.formatDisplayValue(value);
        Ext.ux.senior.form.TriggerTextAreaField.superclass.setValue.call(this, displayValue);
    },
    
    getRawValue : function(){
        return this.originalValue;
    },
    
    /**
     * Recebe como parmetro uma String, substitui as quebras de linha por espaos e trunca a String em 255 caracteres.
     * 
     * @param {String}
     *            value valor que ser formatado
     * @return {String} valor formatado conforme regra do mtodo
     * @private
     */
    formatDisplayValue : function(value) {
        if (value) {
            var subs = ' ';
            if (Ext.isOpera) {
                /*
                 * No Opera uma quebra de linha  considerada como 2 caracteres no posicionamento do Caret.
                 */
                subs = '  ';
            }
            var presentationValue = value.replace(/\n/g, subs);
            presentationValue = presentationValue.replace(/\t/g, ' ');
            /*
             * Segundo requisito, o valor apresentado no edit ser o valor do campo CLOB. Caso este valor exceda 255 caracteres, so apresentados os
             * primeiros 255 caracteres + "..."
             */
            presentationValue = Ext.util.Format.ellipsis(presentationValue, 255, false);

            return presentationValue;
        }
        return '';
    }
});

/**
 * @author Joao.Maas
 */
Ext.ux.senior.form.TriggerTextAreaField.EditorWindow = {

    currentMemoField : null,
    editorWindow : null,
    textArea : null,
    buttonOk : null,
    buttonCancel : null,
    panelButton : null,
    map : null,

    /**
     * @public
     */
    open : function(memoField) {
        this.currentMemoField = memoField;
        this.buildEditorWindow();
        /* Verfica qual o valor do edit */
        var value = memoField.originalValue;
        this.textArea.setValue(value);
        /*Define o componentOwner do text-area para que no ocorra mais o erro de "Connection not found" quando  pressionado tab para mudar o foco.*/
        this.textArea.componentOwner = memoField.id;
        FocusManager.pushContext();
        this.editorWindow.show();
        /*Faz com que o painel com os botes "Ok" e "Cancelar" seja alinhado ao text-area no topo  direita.*/
        this.panelButton.getEl().alignTo(this.textArea.getEl(), "tr");
        var selection = SelectionUtil.getSelection(memoField);
        /* Troca o compValue do selection, pois nesse caso especfico o valor do edit no representa o valor real do componente */
        selection.compValue = value;
        if (selection.selectionStart < 0 && selection.selectionEnd < 0) {
            selection.selectionStart = value.length;
            selection.selectionEnd = value.length;
        }
        this.textArea.selectText(selection.selectionStart, selection.selectionEnd);
        /*
         *  setado um delay de 80 milisegundos no foco da textArea, pois em algumas situaes a tela estava sendo aberta e o foco no ia para o
         * TextArea.
         */
        this.textArea.focus(false, 80);
    },

    /**
     * @private
     */
    buildEditorWindow : function() {
        /*
         * Cria e configura a Janela Modal
         */
        this.editorWindow = new Ext.Window({
            id : "editWindow",
            title : this.currentMemoField.fieldLabel,
            modal : true,
            manager : Ext.ux.senior.WindowMgr,
            layout : 'table',
            layoutConfig : {
                columns : 2
            },
            width : 610,
            height : 300
        });
        /*
         * Cria e configura a TextArea
         */
        this.textArea = new Ext.form.TextArea({
            id : "editTextArea",
            monitorResize : true,
            width : 520,
            height : 260,
            rowspan : 4,
            plugins : [ "componentfocusplugin", "fieldstyle" ],
            focusGroup : "defaultGroup",
            focusOrder : 1
        });

        /*
         * Cria e configura o Boto Ok
         */
        this.buttonOk = new Ext.Button({
            id : "btnOk",
            text : "Ok",
            width : 73,
            plugins : [ "componentfocusplugin" ],
            focusGroup : "defaultGroup",
            focusOrder : 2
        });
        /*
         * Cria e configura o boto Cancel
         */
        this.buttonCancel = new Ext.Button({
            id : "btnCancel",
            text : "Cancelar",
            width : 73,
            plugins : [ "componentfocusplugin" ],
            focusGroup : "defaultGroup",
            focusOrder : 3
        });
        /*
         * Cria e configura o Panel dos Botes
         */
        this.panelButton = new Ext.Panel({
            id : "panelButton",
            layout : 'table',
            layoutConfig : {
                columns : 1
            },
            autoWidth : true,
            autoHeight : true
        });

        this.panelButton.add(this.buttonOk);
        this.panelButton.add(this.buttonCancel);

        /*
         * Posiciona os componentes na Tela
         */
        this.editorWindow.add(this.textArea);
        this.editorWindow.add(this.panelButton);

        /* liga os listeners da tela */
        this.editorWindow.on('beforeclose', this.beforeClose, this);
        this.editorWindow.on('close', this.afterClose, this);
        this.textArea.on('render', this.onTextAreaRender, this);
        this.buttonOk.on('click', this.okClick, this);
        this.buttonCancel.on('click', this.cancelClick, this);
    },

    /**
     * @event textArea#render
     */
    onTextAreaRender : function(ct, position) {
        this.map = new Ext.KeyMap(this.textArea.el, {
            key : Ext.EventObject.SPACE,
            ctrl : true,
            fn : function(e) {
                this.appendTab();
            },
            scope : this
        });
        this.map.stopEvent = true;
    },

    /**
     * adiciona um \t no texto na posio do cursor para simular TAB num texto.
     * 
     * @private
     */
    appendTab : function() {
        var selection = SelectionUtil.getSelection(this.textArea);
        var beforeText = selection.beforeText;
        if (Ext.isIE) {
            /* 
             * Esta verificao  feita pois o calculo do caret no IE tem de ser feito com textRange. Com isso ocorre um problema quando o caret est
             * posicionado no inicio de uma linha, pois a quebra de linha no  includa em nenhum dos textRange. Para corrigir este erro  verificado
             * se logo aps o fim do texto anterior ao caret existe uma quebra de linha. Se existir adiciona esta quebra de linha ao texto anterior ao
             * caret.
             */
            if (this.textArea.getValue().charCodeAt(beforeText.length) == 13) {
                /*
                 * Este if gera um erro quando o usurio aplica uma tabulao no fim de uma linha que no  a ltima do texto. O erro que ocorre  que
                 * a tabulao do usurio tambm cria uma nova linha no texto. Este comportamento ser mantido assim, pois  muito mais comum o
                 * usurio inserir uma tabulao no inicio da linha do que no fim.
                 */
                beforeText = beforeText.concat("\n");
                /*atualizar posio do caret devido a adio da quebra de linha*/
                selection.selectionStart = beforeText.length;
                selection.selectionEnd = beforeText.length + selection.text.length;
            }
        }

        this.textArea.setValue(beforeText + "\t" + selection.afterText);
        /* posicionar o cursor corretamente (logo aps o \t) */
        selection.selectionStart = selection.selectionStart + 1;
        selection.selectionEnd = selection.selectionEnd + 1;
        this.textArea.selectText(selection.selectionStart, selection.selectionEnd);
    },

    /**
     * @event buttonOk#click
     */
    okClick : function() {
    	var value = this.textArea.getValue();
        this.currentMemoField.changeValue(value);
        this.currentMemoField.editText = value;
        this.currentMemoField.setValue(value);
        this.editorWindow.close();
    },

    /**
     * @event buttonCancel#click
     */
    cancelClick : function() {
        this.currentMemoField.editText = this.currentMemoField.originalValue;
        this.currentMemoField.setValue(this.currentMemoField.editText);
        this.editorWindow.close();
    },

    /**
     * @event editorWindow#close
     */
    afterClose : function(panel) {
        this.map.disable();
        FocusManager.popContext();
        FocusManager.focusCurrentComponent();
    },

    beforeClose : function() {
        //garante que a tratativa do blur ser chamada aps a atribuio
        this.currentMemoField.triggerBlur();
        this.currentMemoField.showingModal = false;
    }
};

Ext.reg('seniortextareatrigger', Ext.ux.senior.form.TriggerTextAreaField);