Ext.namespace('Ext.ux.senior');

/**
 * 
 * UC-FRC-0076 - Operar Button.xucs <br>
 * REQ0004 - Ações - O button tem o evento de click, mas para parecer um tipo de menu com outras opções, ou usamos um split no lugar um até um menu mesmo. O problema encontra-se na atualização
 * automática, onde temos que remover o botão do lugar em que está e colocar um split no lugar, isso implica em remover o componente de todo um layout complexo do sistema que deve ser levado em
 * consideração. Recomendo um teste com a propriedade menu. <br>
 * REQ0005 - Imagem do botão - icon, setIcon <br>
 * REQ0006 - Título do botão - text, setText <br>
 * REQ0007 - Estilo do botão - configs: {enableToogle, pressed, toggleGroup, toggleHandler}, methods{toogle} events{toogle} <br>
 * REQ0012 - Habilitar/desabilitar botão - disabled, setDisabled <br>
 * REQ0013 - Alinhamento de ícone no botão - iconAlign, iconCls, setIconClass <br>
 * REQ0014 - Alinhamento de conteúdo no botão - não encontrado nenhuma propriedade para tal <br>
 * REQ0015 - Criar botão - iconbutton suportado pelo button.js, flatbutton - não encontrado nenhum componente semelhante, linkbutton ao invés de ser um botão podemos criar um componente link bem mais
 * simples. <br>
 * REQ0008 - Irrelevante <br>
 * REQ0010 - Irrelevante <br>
 * REQ0001 - Seleção de ação - como recomendado no <br>
 * REQ0004, podemos utilizar um menu associado, o botão faz os alinhamentos automaticamente. <br>
 * REQ0002 - Executar ação - click <br>
 * REQ0003 - Teclas de atalho - por eventos de keydown, keypress, ou usar KeyMap, todos no elemento do botão, o botão em si não tem eventos de teclas<br>
 * REQ0016 - Captura de eventos - evento click do botão<br>
 * 
 */
Ext.ux.senior.Button = Ext.extend(Ext.Button, {

    kind : 'ICONBUTTON',
    
    flatCls : 'senior-btn-flat',

    types : {
        menu : 'MENUBUTTON',
        icon : 'ICONBUTTON',
        flat : 'FLATBUTTON'
    },

    applyStylesToText : true,
    
    /**
     * Teclas de atalho tratadas por este componente
     * 
     * @plugin EventManagerPlugin
     */
    keyMapping : [ Ext.EventObject.DOWN, Ext.EventObject.UP, Ext.EventObject.F4, Ext.EventObject.ENTER, Ext.EventObject.ESC ],

    /**
     * @override
     */
    initComponent : function() {
        Ext.ux.senior.Button.superclass.initComponent.call(this, arguments);
        this.attachMenuEvents();
    },

    setContentAligment : function(align) {
        //É necessário alinhar o item do meio da tabela do botão, que é o que armazena o conteúdo.
        this.getBorders().mc.setStyle('text-align', align);
    },

    /**
     * @override
     */
    onRender : function() {
        Ext.ux.senior.Button.superclass.onRender.apply(this, arguments);

        /* remove o arrow padrão e adiciona o de split se for necessário */
        this.removeButtonArrow();
        this.addButtonArrow();

        if (this.menu) {
            this.menu.zIndex = 25000;
        }

        /* se for do tipo flat */
        if (this.kind == this.types.flat || this.kind == this.types.menu) {
            this.toFlat();
        }

        if (this.applyStylesToText) {
            /* replica os estilos ao div do texto, porque o Button do Ext não surte efeito no texto do botão */
            var textEl = Ext.get("C" + this.id + '-text');
            if (textEl) {
                textEl.applyStyles(this.style);
            }
        }

        if (this.align) {
            this.setContentAligment(this.align);
            delete this.align;
        }
    },

    /**
     * @override
     */
    afterRender : function() {
        Ext.ux.senior.Button.superclass.afterRender.apply(this);
        this.configureBtnSize();
    },

    /**
     * @override
     * @param height a nova altura do botão
     */
    setHeight : function(height) {
        Ext.ux.senior.Button.superclass.setHeight.call(this, height);
        this.height = height;
        this.configureBtnSize();
    },

    /**
     * Configura a altura o elemento button de acordo com a altura total do botão
     * 
     * @privtae
     */
    configureBtnSize : function() {
        // 22 é o tamanho padrão de um botão
        if (this.height && Ext.isNumber(this.height) && this.height != 22) {
            // 6 é a soma da altura dos TRs externos do botão (que dão o efeito de borda arredondada)
            this.btnEl.setStyle('height', (this.height - 6) + 'px');
        }
    },

    /**
     * Pega todos os elementos do botão que utilizam imagens para controle de bordas e fundo
     * 
     * @private
     */
    getBorders : function() {
        if (!this.borders) {
            /* elementos que contém as bordas do botão */
            this.borders = {
                tl : this.el.child('td.x-btn-tl'),
                tc : this.el.child('td.x-btn-tc'),
                tr : this.el.child('td.x-btn-tr'),
                ml : this.el.child('td.x-btn-ml'),
                mc : this.el.child('td.x-btn-mc'),
                mr : this.el.child('td.x-btn-mr'),
                bl : this.el.child('td.x-btn-bl'),
                bc : this.el.child('td.x-btn-bc'),
                br : this.el.child('td.x-btn-br')
            };
        }

        return this.borders;
    },

    /**
     * Transforma o botão em flat
     * 
     * @private
     */
    toFlat : function() {
        /* quando mouse sai de cima ou entra, remove ou adiciona a classe */
        this.on('mouseout', this.onFlatButtonMouseOut);
        this.on('mouseover', this.onFlatButtonMouseOver);
        this.on('toggle', this.onFlatButtonToggle);

        /* já remove no início */
        this.hideBorders();
    },

    onFlatButtonMouseOut : function() {
        this.mouseIsOver = false;

        this.hideBorders();
    },

    onFlatButtonMouseOver : function() {
        this.mouseIsOver = true;

        this.showBorders();
    },

    onFlatButtonToggle : function() {
        //Se o botão foi ativado ou foi desativado e o mouse estiver sobre o botão, precisa mostrar as bordas.
        if (this.pressed || this.mouseIsOver) {
            this.showBorders();
        } else {
            this.hideBorders();
        }
    },

    hideBorders : function() {
        //Se estive pressionado, não esconde as bordas porque o efeito visual de pressionado é feito a partir delas.
        if (this.pressed) {
            return;
        }

        for ( var i in this.getBorders()) {
            this.borders[i].addClass(this.flatCls);
        }
    },

    showBorders : function() {
        for ( var i in this.getBorders()) {
            this.borders[i].removeClass(this.flatCls);
        }
    },

    /**
     * @plugin EventManagerPlugin
     */
    eventMapping : function() {
        var map = {
            itemclick : this.prepareItemClickData
        };

        /*
         * Faço a decisão de como vai ser tratado os eventos de acordo com o tipo de botão
         */

        /* menu não envia click, apenas abre o menu */
        if (this.kind != this.types.menu) {
            map.click = this.processButtonClick;
        }

        return map;
    },

    /**
     * @event itemclick
     */
    prepareItemClickData : function(itemIndex) {
        if (this.blockIntf) {
            this.blockInterface();
        }
        return {
            type : 'clicked',
            target : itemIndex
        };
    },

    /**
     * Bloqueia a inteface
     * 
     * @private
     */
    blockInterface : function() {
        ScreenManager.blockScreen("Processando...", true);
    },

    /**
     * @event click
     */
    processButtonClick : function(cmp, evobj) {
        var t = Ext.get(evobj.target);
        /*
         * Infelizmente o botão gera click mesmo quando o click é na flecha de menu, então temos que fazer este tratamento para quando for click na
         * flecha não envia para o servidor, apenas abre o menu.
         */
        
        // Na execução de testes no Selenium, sempre a posiçãodo mouse é x=0 e y=0, então verifica se é menu apenas pela classe (sem tratamento especial porque nunca ocorrerá clique na borda).
        if (t.is('em.x-btn-split') && !evobj.getPageX()) {
            return null;
        }
        
        if (this.menu) {
            // Se o clique ocorreu é direita do divisor do botão, apenas mostra o menu.
            var visBtn = this.el.child('em.x-btn-split');
            var menuX = visBtn.getRegion().right - visBtn.getPadding('r');
            if (evobj.getPageX() > menuX) {
                return null;   
            }
        }
        
        this.hideMenu();
        var reqObj = {
            type : 'clicked',
            target : -1
        };
        
        if (this.blockIntf) {
            if (evobj.target.className.indexOf('x-btn-split') == -1) {
                this.blockInterface();
            }
        }
        
        return reqObj;
    },

    /**
     * @plugin EventManagerPlugin
     */
    onKeyEvent : function(key, evt) {
        var evs = Ext.EventObject;

        if (this.menu) {
            if (key == evs.F4 || key == evs.DOWN && evs.altKey) {
                if (this.hasVisibleMenu()) {
                    this.hideMenu();
                } else {
                    this.showMenu();
                }
            }
        }

        /* para clicar no botão quando pressionado as teclas abaixo */
        switch (key) {
        case evs.ENTER:
        case evs.SPACE:
            this.fireEvent('click', this, evt);
            break;
        }

    },

    /**
     * Registra o evento de fechar o menu com as teclas F4, ESC, ALT + DOWN e TAB
     * 
     * @private
     */
    registerCloseMenuEvent : function() {

        if (!this.hasVisibleMenu()) {
            return;
        }

        /* cria keymap se ainda não tem, para fechar o menu quando for f4, tab, esc ou alt + down */
        if (!this.menu.keyMap) {

            function closeEvt(e) {
                if (Ext.EventObject.keyCode == Ext.EventObject.DOWN && !Ext.EventObject.altKey) {
                    return;
                }
                this.hideMenu();
                this.menu.keyMap.disable();

                // no safari não troca de abas, fica navegando dentro da tela
                if (Ext.isSafari && Ext.EventObject.keyCode == Ext.EventObject.TAB && e.ctrlKey && e.shiftKey) {
                    return true;
                }

                if (Ext.EventObject.keyCode == Ext.EventObject.TAB) {
                    if (Ext.EventObject.shiftKey) {
                        FocusManager.focusPrevComponent();
                    } else {
                        FocusManager.focusNextComponent();
                    }
                } else {
                    FocusManager.focusCurrentComponent();
                }
            }

            this.menu.keyMap = new Ext.KeyMap(this.menu.getEl(), {
                key : [ Ext.EventObject.F4, Ext.EventObject.TAB, Ext.EventObject.ESC, Ext.EventObject.DOWN ],
                fn : closeEvt,
                scope : this
            });

            /* este keymap é para fazer a seleção do item no space*/
            new Ext.KeyMap(this.menu.getEl(), {
                key : [ Ext.EventObject.SPACE ],
                fn : function() {
                    this.onMenuItemClick(this.menu.activeItem, Ext.EventObject);
                    this.hideMenu();
                    this.menu.keyMap.disable();
                },
                scope : this
            /* neste caso temos que fazer isto, se não o button também vai processar o espaço e vai como click e não como Seleção do item */
            }).stopEvent = true;
            
            
            /* Este keyMap foi separado do de cima pois anteriormente ao precionar a tecla "enter" no menu 
             * estava enviando duas mensagens de click isso ocorria pois o ext já realiza o click ao precionar 
             * a tecla "enter" por padrão. Este keyMap serve apenas para fechar o menu*/
            new Ext.KeyMap(this.menu.getEl(), {
                key : [ Ext.EventObject.ENTER ],
                fn : function() {
                    this.hideMenu();
                    this.menu.keyMap.disable();
                },
                scope : this
            /* neste caso temos que fazer isto, senão o button também vai processar o enter e vai como click e não como seleção do item */
            }).stopEvent = true;

        }

        /* habilita os eventos */
        this.menu.keyMap.enable();
    },

    /**
     * Se este botão tiver um menu, liga um evento de clique para cada item de menu, para este botão tratar
     * 
     * @private
     */
    attachMenuEvents : function() {
        if (this.menu) {
            var items = this.menu.items;
            for ( var i = 0; i < items.getCount(); i++) {
                items.get(i).setHandler(this.onMenuItemClick, this);
            }
        }
    },

    /**
     * Adiciona itens no menu do botão
     * 
     * @public
     */
    updateMenuItems : function(menuConfig) {
        if (this.menu) {
            this.menu.destroy();
        }
        
        if (menuConfig.items.length == 0) {
            /* Itens foram removidos */
            this.removeButtonArrow();
            this.menu = null; // precisa ser feito depois do removeButtonArrow(), que depende do menu
            return;
        }

        menuConfig.xtype = 'menu';
        this.menu = Ext.create(menuConfig);

        this.attachMenuEvents();
        this.addButtonArrow();
    },

    /**
     * Retorna o elemento que contém a seta de menu
     * 
     * @private
     */
    arrowEl : function() {
        if (!this.emEl) {
            this.emEl = this.getEl().child('em');
        }
        return this.emEl;
    },

    /**
     * Simplesmente adiciona a classe de menu (método do próprio button), ao elemento que mantém a seta do menu
     * 
     * @private
     */
    addButtonArrow : function() {
        /* Tipo menu não tem seta */
        if (this.kind == this.types.menu) {
            return;
        }

        /* Só adiciona se tiver menu */
        if (this.menu) {
            var arrowEl = this.arrowEl();
            arrowEl.addClass('x-btn-split');
            arrowEl.addClass(this.getMenuClass());
        }
    },

    /**
     * Simplesmente remoce a classe de menu (método do próprio button), do elemento que mantém a seta do menu
     * 
     * @private
     */
    removeButtonArrow : function() {
        var arrowEl = this.arrowEl();
        arrowEl.removeClass('x-btn-split');
        arrowEl.removeClass(this.getMenuClass());
    },

    /**
     * Quando houver um clique no botão, é disparado o evento itemclick
     * 
     * @event menuitemclick
     */
    onMenuItemClick : function(item, eventObj) {
        this.fireEvent('itemclick', item.index);
    },

    /**
     * @override
     */
    setText : function(text) {
        var newText = this.convertTags(text);
        var id = "C" + this.id + '-text'; /*  id para adicionar estilos posteriormente */
        var htmlText = '<div id=' + id + '>' + newText + '</div>';
        /*  retorna o texto passado como parâmetro para o getText ocorrer corretamente */
        var el = Ext.ux.senior.Button.superclass.setText.call(this, newText ? htmlText : text);
        this.realText = text;
        return el;
    },

    getText : function() {
        return this.realText;
    },

    /**
     * converte os valores < e > para não serem considerados como tags html
     * 
     * @private
     */
    convertTags : function(value) {
        if (!value) {
            return value;
        }
        value = value.replace(/>/g, "&gt;");
        value = value.replace(/</g, "&lt;");
        return value;
    },

    /**
     * Sobrescreve o original para registrar os eventos de tecla e para setar o primeiro item como ativo.
     * 
     * @Override
     */
    showMenu : function() {
        Ext.ux.senior.Button.superclass.showMenu.call(this, arguments);

        Ext.getDoc().on('mousewheel', this.onMouseWheel, this);

        this.registerCloseMenuEvent();
        //feito o defer pois esse método dispara o foco, que altera o eveneto do método processButtonClick
        this.menu.setActiveItem.defer(1, this.menu, [ this.menu.items.get(0), false ]);
    },

    /**
     * Retorna o Elemento que trata o estilo desejado.
     * @param scope  String concatenada com o escopo e a propriedade do estilo desejado.
     */
    getStyleEl : function(scope) {
        var el = null;
        var property = scope.split("$");

        if (property[0] == "button.menu.focused") {
            if (property[1].indexOf("font") > -1) {
                el = this.menu.getEl().child('li.x-menu-item-active span');
            } else {
                el = this.menu.getEl().child('li.x-menu-item-active');
            }

        } else if (property[0] == "button.menu") {
            if (property[1].indexOf("font") > -1) {
                var filtered = this.menu.items.filterBy(function(obj, key) {
                    return !obj.container.hasClass('x-menu-item-active');
                });
                el = filtered.lenght == 0 ? null : filtered.itemAt(0).container.child('span');
            } else {
                el = this.menu.getEl();
            }
        } else {
            el = this.searchElementForButton(scope);
        }

        return el;
    },

    /**
     * Retorna o elemento que trata a propriedade informada.
     * 
     * @private
     */
    searchElementForButton : function(scope) {
        var el = null;
        var property = scope.split("$");
        if (property[1] == "background" || property[1] == 'border-color') {
            el = this.getEl().query('td', false);

        } else {
            el = this.btnEl;
        }

        return el;
    },

    /**
     * Esconde o menu quando acontecer um evento de mousewheel
     * @private
     */
    onMouseWheel : function() {
        this.hideMenu();
    },

    /**
     * Sobrescrito para se desligar do evento de mousewheel.
     * @override
     */
    hideMenu : function() {
        Ext.getDoc().un('mousewheel', this.onMouseWheel, this);
        Ext.ux.senior.Button.superclass.hideMenu.call(this);
    }

});

Ext.reg('seniorbutton', Ext.ux.senior.Button);
