Ext.namespace('Ext.ux.senior.data');

Ext.ux.senior.data.ArrayStore = Ext.extend(Ext.data.ArrayStore, {

    /** contm BOF neste Store */
    atBof : true,
    /** contm EOF neste Store */
    atEof : false,

    /**
     * Define se o Store possui BOF.
     * 
     * @public
     * @param {boolean}
     *            bof se os dados do store possuem BOF
     */
    setBof : function(bof) {
        this.atBof = bof;
    },

    /**
     * Define se o Store possui EOF.
     * 
     * @public
     * @param {boolean}
     *            eof se os dados do store possuem EOF
     */
    setEof : function(eof) {
        this.atEof = eof;
    },

    /**
     * Retorna se o Store possui BOF.
     * 
     * @public
     * @return {boolean} true se o store possui BOF e false se no possuir
     */
    isAtBof : function() {
        return this.atBof;
    },

    /**
     * Retorna se o Store possui EOF.
     * 
     * @public
     * @return {boolean} true se o store possui EOF e false se no possuir
     */
    isAtEof : function() {
        return this.atEof;
    },

    /**
     * Remove os registros do Store a partir do recordKey.
     * 
     * @public
     * @param {Ext.data.Record}
     *          record registro que ser removido
     */
    removeById : function(record) {
        if (Ext.isArray(record)) {
            Ext.each(record, function(recordKey) {
                this.remove(this.getById(recordKey));
            }, this);
        } else {
            this.remove(this.getById(record));
        }
    },

    /**
     * Cria um registro que pode ser inserido neste store.
     * 
     * @public
     * @param {Array} rawData Array contendo os dados do registro que deve ser criado
     * @returns {Array} o registro criado
     */
    createRecord : function(rawData) {
        return this.reader.readRecords(rawData).records;
    },

    /**
     * Insere o registro no store a partir da posio indicada.
     * 
     * @public
     * @param position posio que o registro ser inserido no store
     * @param records Registro que ser inserido
     * @param isRecordNew Flag indicando se o registro  novo
     * @param isDirty se o registro est alterado
     * @param dirtyFields Campos a marcar como alterados (doritos)
     * @param hasErrors se o registro possui erros
     * @param errorFields campos que possuem erros
     */
    insertRecord : function(position, records, isRecordNew, isDirty, dirtyFields, hasErrors, errorFields) {
        this.insert(position, records);
        this.renderRecordMarks(position, isRecordNew, isDirty, dirtyFields, hasErrors, errorFields);
    },

    /**
     * Cria o registro a partir do <code>rawData</code> e insere o registro no store a partir da posio indicada.
     * @public
     * @param position posio que o registro ser inserido no store
     * @param rawData Array contendo os dados do registro que deve ser criado
     * @param isRecordNew Flag indicando se o registro  novo
     * @param isDirty se o registro est alterado
     * @param dirtyFields Campos a marcar como alterados (doritos)
     * @param hasErrors se o registro possui erros
     * @params errorFields campos que possuiem erros
     */
    createAndInsertRecord : function(position, rawData, isRecordNew, isDirty, dirtyFields, hasErrors, errorFields) {
        this.insertRecord(position, this.createRecord(rawData), isRecordNew, isDirty, dirtyFields, hasErrors, errorFields);
    },

    /**
     * Marca os campos do registro informado como alterados.
     * 
     * @public
     * @param {int}
     *      index ndice do registro
     * @param {String}
     *      fields Campos que esto alterados
     */
    markDirty : function(index, fields) {
        var record = this.getAt(index);
        record.dirty = true;
        if (!record.modified) {
            record.modified = {};
        }
        for ( var i = 0; i < fields.length; i++) {
            record.modified[fields[i]] = true;
        }
    },

    /**
     * Marca o campo como um registro novo (com a marcao de doritos).
     * 
     * @public
     * @param {int}
     *      index ndice do registro marcado como novo
     */
    markNew : function(index) {
        var record = this.getAt(index);
        record.phantom = true;
    },

    markError : function(index, hasErrors, errorFields) {
        var record = this.getAt(index);
        record.error = hasErrors;
        record.errorFields = errorFields;
    },

    /**
     * Limpa os campos modificados, flag de registro alterado e de registro novo.
     * 
     * @param index ndice do registro que ter suas flags limpas.
     */
    resetChanges : function(index) {
        var record = this.getAt(index);
        if (record) {
            record.phantom = false;
            record.dirty = false;
            if (record.modified) {
                delete record.modified;
            }
        }
    },

    /**
     * Mtodo que retorna um registro pelo <code>id</code> e se este no existir, cria e adiciona no Store. 
     * No deve ser utilizado normalmente, pois pode deixar o Store inconsistente. As chamadas normalmente viro do servidor que est atualizando um registro que ainda no estava no client.   
     * @param id Identificador do regitro
     * @returns O registro com o id correspondente
     */
    safeGetById : function(id) {
        var record = this.getById(id);
        if (!record) {
            record = new this.recordType({}, id);
            this.add([ record ]);
        }
        return record;
    },

    /**
     * Trata do update dos dados de um nico registro.<br/><br/>
     * 
     * Quando um registro  atualizado na grid,  realizada apenas a alterao de seus valores, no entanto, <br/>
     * caso o registro tenha seu ndice alterado na grid durante a alterao dos dados(keepDataOrder = false), <br/>
     *  necessrio remover o registro e adicion-lo novamente. 
     * 
     * @public
     * @param id Identificador do registro
     * @param fieldNames Nomes dos campos do registro alterado
     * @param values Valores do registro alterado
     * @param index ndice em que se encontra o registro
     * @param isRecordNew Se o registro  novo
     * @param isDirty se o registro est sujo
     * @param dirtyFields Campos a marcar como alterados (doritos)
     * @param hasErrors se o registro possui erros
     * @param errorFields Campos com erro (cobrinha)
     */
    updateRecord : function(id, fieldNames, values, index, isRecordNew, isDirty, dirtyFields, hasErrors, errorFields) {
        var record;
        if (this.indexOfId(id) == index) {
            record = this.safeGetById(id);
            for ( var i = 0; i < fieldNames.length; i++) {
                record.data[fieldNames[i]] = values[i];
            }
            this.renderRecordMarks(index, isRecordNew, isDirty, dirtyFields, hasErrors, errorFields);
            record.afterEdit();
        } else {
            record = this.getById(id);
            for ( var i = 0; i < fieldNames.length; i++) {
                record.data[fieldNames[i]] = values[i];
            }
            this.removeById(id);
            this.suspendEvents(false);
            this.insert(index, record);
            this.renderRecordMarks(index, isRecordNew, isDirty, dirtyFields, hasErrors, errorFields);
            this.resumeEvents();
            this.fireEvent("add", this, [ record ], index);
            this.fireEvent("updateEditor", this, index, [ record ]);
        }
    },

    /**
     * Adiciona as marcaes necessrias a um registro. (Marcao de "novo registro" e "doritos"). 
     * 
     * @public
     * @param index ndice do registro a ser marcado
     * @param isRecordNew Se o registro  novo
     * @param isDirty Se o registro foi alterado
     * @param dirtyFields Campos a marcar como alterados (doritos)
     * @param hasErros Seo registro possui erros
     * @param errorFields Campos com erro (cobrinha)
     */
    renderRecordMarks : function(index, isRecordNew, isDirty, dirtyFields, hasErrors, errorFields) {
        if (isRecordNew) {
            this.markNew(index);
        }
        if (isDirty) {
            this.markDirty(index, dirtyFields);
        }
        this.markError(index, hasErrors, errorFields);
    },

    /**
     * Manda marcar os campos de um registro que ainda no foi alterado (ainda no existe no servidor) com erro.
     * @param id o id do registro
     * @param index o ndice do registro com erro
	 * @param hasError indica se o registro tem ou no erros
     * @param errorFields os campos do registro que esto com erro.
     */
    setErrorRecord : function(id, index, hasError, errorFields) {
        var record = this.getById(id);
        this.updateRecord(id, record.fields, record.data, index, record.phantom, record.dirty, record.modified, hasError, errorFields);
    }

});

Ext.reg('seniorarraystore', Ext.ux.senior.data.ArrayStore);
