Ext.namespace('Ext.ux.senior.grid');

Ext.ux.senior.grid.EditorGridPanel = Ext.extend(Ext.grid.EditorGridPanel, {

    rowOver : null,

    clicksToEdit : 1,

    deferRowRender : false,

    gridHints : null,

    dataLoaded : false,

    /**
     * Sufixo para as colunas de rawvalue
     */
    RAW_VALUE_SUFFIX : "$rawValue",

    /* elemento da pgina utilizado para 'fakear' o evento de blur para a grid */
    doc : null,

    alwaysHighlightFocused : true,

    /**
     * Define o estilo visual inicial da grid. 
     */
    customizations : null,

    /**
     * Armazena o estado do checkbox do header da coluna. 
     */
    headerCheckBoxChecked : false,

    /**
     * Customizaes para clula/linha focada
     */
    focusCustomizations : null,

    /**
     * Sobrescrito o valor padro (25) para que sempre haja espao para o ellipsis ao redimensionar as colunas.
     */
    minColumnWidth : 40,
    
    /**
     * @cfg {Boolean} allowDelete Flag que define se uma grid pode deletar registros
     */

    /**
     * @cfg {Boolean} allowInsert Flag que define se uma grid pode inserir registros
     */

    /**
     * @cfg {String} columnsEditabilityFieldName Nome do campo que cada registro desta grid ter, cujo o valor informa quais campos (colunas) podem ou no ser editados.
     */

    /**
     * @cfg {Object} customizations Objeto que armazena as customizaes que devem ser aplicados logo aps a renderizao da view da grid.<br>
     *      O objeto  formado pelas seguintes propriedades:<br>
     *      colCustomizations: Objeto cujo o atributo refere-se ao nome da coluna e o valor ao objeto de customizao daquela coluna.<br>
     *      <br>
     *      rowCustomizations: Objeto cujo o atributo refere-se ao nmero da linha e o valor ao objeto agrupador de customizao daquela linha.<br>
     *      O objeto agrupador de customizao da linha  composto por duas propriedades:<br>
     *      rowCustomization: Objeto de customizao da linha.<br>
     *      cellCustomizations: Objeto cujo o atributo refere-se ao nome da coluna e o valor ao objeto de customizao para aquela clula (linha/coluna).<br>
     *      <br>
     *      O objeto de customizao composto por duas propriedades:<br>
     *      style: Estilo customizado. hints: Hints customizados. <br>
     *      Se algum valor no foi definido, significa que aquela linha/coluna/clula no possui estilo customizado. <br>
     *      <br>
     *      <br>
     *      Exemplo:
     * 
     * <pre>
     * customizations: {
     *      colCustomizations: {
     *          nome: {
     *              hints: ['Nome Colaborador']
     *          }
     *      },
     *      rowCustomizations: {
     *          0: {
     *              rowCustomization: {
     *                  style: 'estilo'
     *              },
     *              cellCustomizations: {
     *              idade: {
     *                  style: 'estilo2',
     *                  hints: ['Idade Colaborador']
     *                  }
     *              }
     *          }
     *      }
     * }
     * </pre>
     */

    /**
     * @cfg {Number} initialVisibleRow Linha que deve estar aparecendo na grid quando ela for renderizada.
     */

    initComponent : function() {
        if (!this.columnsEditabilityFieldName) {
            throw "The required property \"columnsEditabilityFieldName\" was not set.";
        }

        Ext.ux.senior.grid.EditorGridPanel.superclass.initComponent.call(this, arguments);
        this.addEvents('scroll', 'showfocused');

        /** @see Observable#relayEvents */
        this.relayEvents(this.view, [ 'sortcolumn' ]);

        /*
         * A grid tem eventos dinamicos, quando tratamos os eventos genricos em cima de toda a grid como, click na grid, over, double click, keydown,
         * keypress, ou qualquer outro evento basico, que no diz respeito se  no header, na linha ou enfim, o process event far isto para ns.
         * 
         * Ento se o over for em cima do header, a grid vai disparar o evento: headermouseover
         * 
         * Se for em cima da linha: rowmouseover Se for fora da linha: rowmouseout
         * 
         * Para o momento s precisamos do rowmouseover e rowmouseout, no qual vir os seguintes parmetros:
         * 
         * rowmouseover(Grid this, Number rowIndex, Ext.EventObject e)
         * 
         */
        this.on('mouseover', function(evt) {
            /* como um process event dispara o evento "raw" tambm, ento temos que fazer este controle para no entrar em recurso infinita */
            if (!this.processingMouseOver) {
                this.processingMouseOver = true;
                this.processEvent('mouseover', evt);
                this.processingMouseOver = false;
                return false;
            }
            return true;
        }, this);

        this.on('mouseout', function(evt) {
            /* como um process event dispara o evento "raw" tambm, ento temos que fazer este controle para no entrar em recurso infinita */
            if (!this.processingMouseOut) {
                this.processingMouseOut = true;
                this.processEvent('mouseout', evt);
                this.processingMouseOut = false;
                return false;
            }
            return true;
        }, this);

        this.on('beforeedit', this.onBeforeEdit);
        this.on('keydown', this.onKeyDown);
        this.on('mousedown', this.onGridMouseDown);
        this.on('resize', this.view.onResize);
        this.getView().on('refresh', this.onViewRefresh, this);
        this.getView().on('realignEditor', this.rowRemovedRealignEditor);
        this.getView().on('afterrender', this.onViewAfterRender);

    },

    rowRemovedRealignEditor : function(view, rowIndex, record) {
        if (this.grid.editing && this.grid.activeEditor) {
            this.grid.activeEditor.realign(false);
        }

    },

    /**
     * @override
     */
    onDestroy : function() {
        /* Ir entrar aqui somente se o componente foi renderizado.
         * Ento a varivel "this.doc" estar com seu valor setado.
         */
        if (this.doc) {
            this.doc.un('mousedown', this.mimicBlur, this);
        }
        Ext.ux.senior.grid.EditorGridPanel.superclass.onDestroy.apply(this, arguments);
    },

    /**
     * @override
     */
    onRender : function() {

        this.doc = Ext.isIE ? Ext.getBody() : Ext.getDoc();

        Ext.ux.senior.grid.EditorGridPanel.superclass.onRender.apply(this, arguments);

        this.doc.on('mousedown', this.mimicBlur, this);
    },

    onViewAfterRender : function() {
        /*
         * as colunas que so resizable == false so renderizadas com a propriedade fixed == true, 
         * para que o seu tamanho no seja alterado caso a grid seja autoFill. Depois da renderizao devem voltar a ser fixed == false.
         */
        var columns = this.getColumnModel().columns;
        for ( var i = 0; i < columns.length; i++) {
            if (columns[i].id != 'statuscolumn' && columns[i].id != 'deletecolumn' && columns[i].id != 'checker') {
                columns[i].fixed = false;
            }
        }

    },

    /* simula um blur para a grid */
    mimicBlur : function(evt, el, o) {
        var nonGridFocus = !this.getGridEl().contains(evt.target);
        if (nonGridFocus) {
            this.onBlur(this);
        }
    },

    /**
     * Mtodo executado quando ocorre um blur na grid.
     * @param grid
     * @public
     * @returns
     */
    onBlur : function(grid) {
        this.showHighlightFocused(this.alwaysHighlightFocused);
    },

    /*
     * Indica se a grid ir exibir ou se ir esconder o highlight da grid.
     * @private
     * @param show true para exibir e false para esconder o highlight
     */
    showHighlightFocused : function(show) {
        var selection = this.getSelectionModel().getSelectedCell ? this.getSelectionModel().getSelectedCell() : null;
        if (selection && selection.length > 0) {
            if (show) {
                this.getView().canShowHighlightFocused = true;
                this.getView().onCellSelect(selection[0], selection[1]);
            } else {
                this.getView().canShowHighlightFocused = false;
                this.getView().onCellDeselect(selection[0], selection[1]);
            }

        }
    },

    onViewRefresh : function(view) {
        this.getColumnModel().getColumnById('statuscolumn').refreshStatus();
    },

    onGridMouseDown : function(e) {
        if (!this.activeEditor) {
            e.preventDefault();
            this.focus(false, false);
            FocusManager.onComponentFocus(this);
        }
    },

    notifyRowOver : function(grid, index, evt) {
        this.rowOver = index;
    },

    getRowOver : function() {
        return this.rowOver;
    },

    eventMapping : function() {
        return {
            scroll : this.onScroll,
            showFocused : this.onShowFocusedClick,
            selection : this.onSelectRow,
            headerclick : this.onHeaderClick,
            sortcolumn : this.onSortColumn,
            deleteRow : this.onDeleteRow,
            insertRow : this.onInsertRow,
            cellclick : this.onCellClick,
            celldblclick : this.onCellDoubleClick,
            resize : this.onResizeGrid,
            columnFixed : this.columnFixed,
            checkBoxClick : this.onHeaderCheckBoxClick,

            columnresize : this.onColumnResize,
            viewready : this.onViewReady,
            columnChangePosition : this.onColumnChangePosition,
            trydeletefasteditrecord : this.onTryDeleteFastEditRecord,
            tryinsertfasteditrecord : this.onTryInsertFastEditRecord
        };
    },

    onResizeGrid : function(adjWidth, adjHeight, rawWidth, rawHeight) {
        this.view.calculateVisibleRows();
        return {
            type : "reloadData",
            pageSize : this.view.visibleRows
        };
    },

    onColumnChangePosition : function(col, newIndex) {
        if (this.editing) {
            this.stopEditing(true);
        }

        return {
            type : 'columnChangePosition',
            columnName : col,
            newColumnPosition : newIndex
        };

    },

    onCellDoubleClick : function(grid, row, column, event) {
        var xtypeColumn = this.getColumnModel().getColumnAt(column).xtype;
        if (xtypeColumn === undefined || xtypeColumn === 'statuscolumn' || xtypeColumn === 'deletecolumn') {/*  se for undefined  checkboxcolumn */
            return;
        }
        var record = this.getStore().getAt(row);
        return {
            type : 'doubleclick',
            rowIndex : row,
            columnIndex : this.getColumnModel().getDataIndex(column),
            isDoubleClick : true,
            recordId : record.id
        };
    },

    /**
     * Pede para o servidor posicionar no registro focado.
     */
    onShowFocusedClick : function() {
        return {
            type : 'showfocused'
        };
    },

    onViewReady : function() {
        this.view.onViewReady.call(this.view);

        if (!this.dataLoaded) {
            this.dataLoaded = true;
            return {
                type : "loadData",
                pageSize : this.view.visibleRows
            };
        }

        /* Aplica os estilos customizados . */
        if (this.customizations) {
            this.applyCustomizations();
        }

        /* Mostra a linha configurada para aparecer inicialmente */
        if (this.initialVisibleRow) {
            this.showRecord(this.initialVisibleRow);

            delete this.initialVisibleRow;
        }
    },

    /**
     * Disparado pelo GridView quando  necessrio enviar ao servidor um evento de scroll.
     */
    onScroll : function(firstVisibleKey, focusedKey, visibleRows, event, focus) {
        var firstRecordKey = this.getStore().getAt(0).data.recordKey;
        return {
            type : 'scroll',
            firstRecord : firstRecordKey,
            indexActive : firstVisibleKey,
            focused : focusedKey,
            visibleRows : visibleRows,
            scrollEvent : event,
            focusRecord : focus
        };
    },

    /**
     * Envia ao servidor um evento de seleo ou focus na grid.
     * 
     * @param {Ext.Grid} grid o gridpanel.
     * @param {String} eventType 'focusRow' para focar uma linha ou 'selectRow' para selecionar uma linha.
     * @param {String} typeSelection tipo do evento que ocorreu. Pode ser ROW (clique em uma linha), NEXT ou PREVIOUS.
     * @param {String} selection o tipo de seleo. Pode ser SINGLE, MULTI, INTERVAL, ALL ou CLEAR.
     * @param {Number} row a linha que recebeu a seleo (opcional)
     */
    onSelectRow : function(grid, eventType, typeSelection, selection, row) {
        return {
            type : eventType,
            typeSelection : typeSelection,
            selection : selection,
            row : row > -1 ? this.view.ds.getAt(row).data.recordKey : row
        };
    },

    focusRow : function(row, forceHtmlFocus) {
        if (!this.editing) {
            this.getSelectionModel().focusRow(row, forceHtmlFocus);
        }
    },

    columnFixed : function(field, fixed) {
        return {
            type : 'columnFixed',
            field : field,
            fixed : fixed
        };
    },

    onHeaderClick : function(grid, columnIndex) {
        if (!this.getColumnModel().isSortable(columnIndex)) {
            return;
        }

        /*
         * Tarefa 224788: Ajuste para desaparecer editor da grid  
         */
        if (this.editing) {
            this.stopEditing();
        }

        return {
            type : 'toggleSortField',
            field : grid.getColumnModel().getDataIndex(columnIndex),
            /* Para no deixar a grid ordenar por ela mesma */
            cancel : true
        };

    },

    /**
     * Quando o a coluna for ordenada pelas opes de asc ou desc
     * 
     * @param colIndex {int} ndice da coluna
     * @param order {String} 'asc' ou 'desc' para ascendente ou descendente
     * @param evt {EventObject} o evento
     * 
     * @event sortcolumn
     */
    onSortColumn : function(colIndex, order) {
        /*
         * Tarefa 224788: Ajuste para desaparecer editor da grid  
         */
        if (this.editing) {
            this.stopEditing();
        }

        /* Pega o nome do campo */
        var field = this.getColumnModel().getDataIndex(colIndex);

        /* monta o objeto de request */
        return {
            type : 'configureSortField',
            direction : order,
            field : field
        };

    },

    onDeleteRow : function(rowIndex) {
        return {
            type : 'deleteRecord',
            index : rowIndex,
            cancel : true
        };
    },

    /**
     * Envia para o servidor o estado do checkbox do header da coluna
     * @param checked valor do check do checkbox
     */
    onHeaderCheckBoxClick : function(col, checked) {
        return {
            type : 'columnCheckboxClicked',
            field : this.getColumnModel().getDataIndex(col),
            checked : checked

        };
    },

    onInsertRow : function(insertAtEof) {
        return {
            type : 'insertRecord',
            value : insertAtEof ? true : false
        };
    },

    /**
     * Executa validaes adicionais para verificar se a clula pode ser editada. A verificao de se a coluna  editvel j  feita antes de chamar esta funo.
     */
    onBeforeEdit : function(editObj) {
        /*
         * As condies para que uma clula da grid possa ser editada so as seguintes:
         *  - Se o campo que a coluna que a clula pertence pode ser editado (esta validao  feita pela prpria grid, antes de chegar nesta funo).
         *  - Se a grid no est readOnly;
         *  - Se a clula em si pode ser editada (esta validao  feita atravs do campo de editabilidade das colunas, presente em cada registro da grid).
         *      - Esta validao assume que por padro todas as clulas podem ser editadas. S sero consideradas no editveis aquelas declaradas no campo de editabilidade de colunas.
         */

        if (this.readOnly) {
            return false;
        }

        return this.isCellEditable(editObj.record, editObj.field);
    },

    isCellEditable : function(record, field) {
        var columnsEditability = record.get(this.columnsEditabilityFieldName);
        /*  Se nem tiver a editabilidade das colunas definidas,  pq todas podem ser editadas. */
        if (!columnsEditability) {
            return true;
        }
        /*  Retorna se a coluna pode ser editada. */
        var editable = columnsEditability[field];
        /*S retorna "false" se a editabilidade estiver configurada false, caso esteja configurada como true ou no esteja definida retorna "true"*/
        return editable !== false;
    },

    /*TODO verificar sobre o parmetro ser colIndex ao invs de fieldName */
    isColumnEditable : function(record, colIndex) {
        if (this.readOnly) {
            return false;
        }

        var colConfig = this.getColumnModel().getColumnAt(colIndex);
        if (!colConfig.editable) {
            return false;
        }

        return this.isCellEditable(record, colConfig.dataIndex);
    },

    /**
     * Sobrescreve o mtodo do Ext para que aps a edio seja restaurado o componente focado. Isto atende o caso em que o editor est ativo e  feito um click diretamente em outro componente.
     * 
     * @override
     */
    onEditComplete : function(ed, value, startValue) {
        Ext.ux.senior.grid.EditorGridPanel.superclass.onEditComplete.call(this, ed, value, startValue);
        this.afterEditCompleteSetRawValue(ed, value);
        /*  faz com defer pois o foco da clula aps a edio tambm roda com defer */
        FocusManager.focusCurrentComponent.defer(1, FocusManager);
    },

    /**
     * Aps a edio seta no store o mesmo valor no rawValue. porque quando acontece um erro o server no atualiza o store
     * 
     * @private
     * @param editor editor
     * @param value Valor 
     */
    afterEditCompleteSetRawValue : function(editor, value) {
        var record = editor.record;
        var field = this.colModel.getDataIndex(editor.col);
        if (record.fields.get(field + this.RAW_VALUE_SUFFIX) != null) {
            record.set(field + this.RAW_VALUE_SUFFIX, value);
        }
    },

    /**
     * Evento de resize de uma coluna.
     * @param colIndex ndice da coluna que sofreu resize
     * @param newSize novo tamanho da coluna
     * @returns objeto de request para VScrollGrid
     */
    onColumnResize : function(colIndex, newSize) {
        this.view.updateHeaderTooltip(colIndex);

        var column = this.getColumnModel().getColumnAt(colIndex).name;

        return {
            type : 'columnResize',
            columnName : column,
            size : Math.floor(newSize) // quando o zoom est diferente de 100% no IE, o novo tamanho calculado pelo Ext pode conter uma parte decimal
        };
    },

    setReadOnly : function(value) {
        this.readOnly = value;
        if (this.readOnly && this.editing) {
            this.stopEditing();
        }
    },

    /**
     * 
     * @public
     */
    setColumnFixed : function(field, value) {
        var cm = this.getColumnModel();
        var col = cm.findColumnIndex(field);
        cm.setLocked(col, value);
    },

    setColumnSortable : function(field, value) {
        var cm = this.getColumnModel();
        var index = cm.findColumnIndex(field);
        cm.setSortable(index, value);
    },

    /**
     * Atribui o valor do checkBox do header da coluna.
     * @param index o indice da coluna que vai receber o check
     * @param value true para checado e false para no checado
     */
    setHeaderCheckBoxChecked : function(field, value) {
        var cm = this.getColumnModel();
        var index = cm.findColumnIndex(field);
        cm.setHeaderCheckBoxChecked(index, value);
    },

    /**
     * Retorna o valor do estado checkbox do header da coluna;
     * @returns
     */
    getHeaderCheckBoxChecked : function() {
        return this.getColumnModel().getHeaderCheckBoxChecked();
    },

    /**
     * Define se a coluna da grid vai ser redimensionavel ou no;
     * @param field campo da grid que ser atribuido o novo valor
     * @param value true se pode ser expansiva e false caso no pode 
     * @returns
     */
    setColumnResizable : function(field, value) {
        var cm = this.getColumnModel();
        var index = cm.findColumnIndex(field);
        cm.setResizable(index, value);
    },

    onTryInsertFastEditRecord : function() {
        return {
            type : 'tryInsertFastEditRecord'
        };
    },

    onTryDeleteFastEditRecord : function() {
        return {
            type : 'tryDeleteFastEditRecord'
        };
    },

    onKeyDown : function(event) {

        var key = event.keyCode, ctrl = event.ctrlKey, shift = event.shiftKey;
        var editable = !this.readOnly;
        var E = Ext.EventObject;

        switch (key) {
        case E.DELETE:
            if (ctrl && editable && this.getStore().getCount() > 0) {
                this.fireEvent("deleteRow", -1);
                return;
            }
            break;
        case E.TAB:
            if (ctrl) {
                /* FocusManager.focusNext(). */
                return;
            }
            break;

        case E.UP:
        case E.DOWN:
            /*  this.view.onDataNavigate(); */
            break;
        }

        var sm = this.getSelectionModel();
        if (sm && sm.onKeyDown) {
            return sm.onKeyDown(event, key, ctrl, shift);
        }
    },

    focus : function(selectText, delay) {
        this.showHighlightFocused(true);
        if (FocusManager.getFocusedComponent() === this) {
            return;/*se j  o focado, no faz nada*/
        }
        if (delay) {
            this.focusTask = new Ext.util.DelayedTask(this.focus, this, [ selectText, false ]);
            this.focusTask.delay(Ext.isNumber(delay) ? delay : 10);
            return;
        }
        if (this.rendered && !this.isDestroyed) {
            var button = Ext.EventObject.button;
            this.getView().focusEl.focus();
            //Se a grid possuir registros e no houver nenhuma clula focada, foca a primeira clula visvel.
            if (this.store.getCount() != 0 && !this.selModel.hasSelection()) {
                this.selModel.selectFirstRecord(this);
            }
            Ext.EventObject.button = button;
            /*  chama explicitamente porque na grid no  disparado naturalmente  */
            FocusManager.onComponentFocus(this);

            /* Se estiver editando, repassa o foco para o editor ativo */
            if (this.editing && this.activeEditor) {
                this.activeEditor.focus();
            }

        }
        return this;
    },

    /**
     * Fora a grid a mostrar o registro referenciado pelo "row" e ajusta o scroll se for necessrio.
     * 
     * @public
     */
    showRecord : function(row) {
        this.getView().stopEvents();
        this.getView().showRecord(row);
        this.getView().resumeEvents();
    },

    setCustomizations : function(customization, focusCustomization) {
        this.customizations = customization;
        this.focusCustomizations = focusCustomization;
        this.applyCustomizations();
        this.view.updateHighlightFocused();
    },

    /**
     * Remove o estilo customizado da clula com as coordenadas passadas em <i>rowIndex</i> e <i>colIndex</i>.
     * Se o ndice da coluna no for informado ser removido o estilo customizado de todas as clulas da linha informada.
     * Se o ndice da linha no for informado, a funo no far nada. 
     * 
     * @param rowIndex ndice da linha.
     * @param colIndex ndice da coluna (opcional)
     */
    removeCustomStyle : function(rowIndex, colIndex) {
        if (!Ext.isDefined(rowIndex)) {
            return;
        }
        var customization = this.customizations;

        if (customization == undefined || customization == null) {
            return;
        }

        var cm = this.getColumnModel();
        var record = this.getStore().getAt(rowIndex);

        if (record == undefined || record == null) {
            return;
        }

        var colCustomizations = customization.colCustomizations;
        var rowCustomizations = customization.rowCustomizations;
        var rowCustomizationRecord = rowCustomizations ? rowCustomizations[record.id] : undefined;
        var rowCustomizationObjRecord = null;
        var cellCustomizationsRecord = null;
        if (rowCustomizationRecord) {
            cellCustomizationsRecord = rowCustomizationRecord.cellCustomizations;
            rowCustomizationObjRecord = rowCustomizationRecord.rowCustomization;
        } else if (!colCustomizations) {
            return;
        }
        if (colIndex) {
            var fieldName = this.getColumnModel().getDataIndex(colIndex);
            //caso o fieldName seja igual a '0' ento  uma coluna especial (coluna de Status ou de Delete) ento  necessrio buscar a clula pelo indice da linha e coluna
            var cellElement = fieldName != '0' ? this.getCellByRecord(record, fieldName) : this.getCellByIndex(rowIndex, colIndex);
            this.removeCustomStyleClass(cellElement, cellCustomizationsRecord, colCustomizations, rowCustomizationObjRecord, fieldName);
        } else {
            for ( var col = 0, colLen = cm.getColumnCount(); col < colLen; col++) {
                var fieldName = this.getColumnModel().getDataIndex(col);
                var cellElement = fieldName != '0' ? this.getCellByRecord(record, fieldName) : this.getCellByIndex(rowIndex, col);
                this.removeCustomStyleClass(cellElement, cellCustomizationsRecord, colCustomizations, rowCustomizationObjRecord, fieldName);
            }
        }
    },
    /**
     * Remove o estilo customizado de uma clula especfica.
     * @param cellElement elemento que representa a clula onde ser removido o estilo customizado
     * @param cellCustomizations um array contendo as customizaes das clulas da linha passada em <i>rowIndex</i>.
     * @param colCustomizations um array contendo as customizaes de todas as colunas
     * @param rowCustomizationObj o estilo customizado da linha passada em <i>rowIndex</i>
     * @param fieldName nome da coluna atual
     * @private
     */
    removeCustomStyleClass : function(cellElement, cellCustomizations, colCustomizations, rowCustomizationObj, fieldName) {

        var cellCustomizationObj = cellCustomizations ? cellCustomizations[fieldName] : undefined;
        if (cellCustomizationObj && cellCustomizationObj.style) {
            cellElement.removeClass(cellCustomizationObj.style);
        } else {
            var colCustomizationObj = colCustomizations ? colCustomizations[fieldName] : undefined;
            if (colCustomizationObj && colCustomizationObj.style) {
                cellElement.removeClass(colCustomizationObj.style);
            } else if (rowCustomizationObj && rowCustomizationObj.style) {
                cellElement.removeClass(rowCustomizationObj.style);
            }
        }
    },

    /**
     * Reaplica o estilo customizado em uma clula especfica da grid ou em todas as clulas da linha especificada em <i>row</i> se
     * <i>col</i> no for informado.
     * @param row ndice da linha.
     * @param col ndice da coluna (opcional)
     */
    restoreCustomStyle : function(row, column) {
        if (!Ext.isDefined(row)) {
            return;
        }

        var customization = this.customizations;

        if (customization == undefined || customization == null) {
            return;
        }

        var cm = this.getColumnModel();

        var record = this.getStore().getAt(row);
        if (this.getSelectionModel().getSelected) {
            record = this.getSelectionModel().getSelected();
        }

        if (record == undefined || record == null) {
            return;
        }

        var colCustomizations = customization.colCustomizations;
        var rowCustomizations = customization.rowCustomizations;

        var rowCustomization = rowCustomizations ? rowCustomizations[record.id] : undefined;
        var rowCustomizationObj = null;
        var cellCustomizations = null;
        if (rowCustomization) {
            cellCustomizations = rowCustomization.cellCustomizations;
            rowCustomizationObj = rowCustomization.rowCustomization;
        } else if (!colCustomizations) {
            return;
        }

        if (column) {
            var fieldName = this.getColumnModel().getDataIndex(column);
            var cellElement = fieldName != '0' ? this.getCell(fieldName, row) : this.getCellByIndex(row, column);
            this.applyCustomStyleClass(cellElement, cellCustomizations, colCustomizations, rowCustomizationObj, fieldName);

        } else {
            for ( var col = 0, colLen = cm.getColumnCount(); col < colLen; col++) {
                var fieldName = this.getColumnModel().getDataIndex(col);
                var cellElement = fieldName != '0' ? this.getCell(fieldName, row) : this.getCellByIndex(row, col);
                this.applyCustomStyleClass(cellElement, cellCustomizations, colCustomizations, rowCustomizationObj, fieldName);
            }
        }

    },

    /**
     * Aplica o estilo customizado em uma clula especfica.
     * @param cellElement elemento que representa a clula onde ser aplicado o estilo customizado
     * @param cellCustomizations um array contendo as customizaes das clulas da linha passada em <i>rowIndex</i>.
     * @param colCustomizations um array contendo as customizaes de todas as colunas
     * @param rowCustomizationObj o estilo customizado da linha passada em <i>rowIndex</i>
     * @param fieldName nome da coluna atual
     * @private
     */

    applyCustomStyleClass : function(cellElement, cellCustomizations, colCustomizations, rowCustomizationObj, fieldName) {

        if (this.canApplyCustomStyle(cellElement)) {

            var cellCustomizationObj = cellCustomizations ? cellCustomizations[fieldName] : undefined;
            if (cellCustomizationObj && cellCustomizationObj.style) {
                cellElement.addClass(cellCustomizationObj.style);
            } else {
                var colCustomizationObj = colCustomizations ? colCustomizations[fieldName] : undefined;
                if (colCustomizationObj && colCustomizationObj.style) {
                    cellElement.addClass(colCustomizationObj.style);
                } else if (rowCustomizationObj && rowCustomizationObj.style) {
                    cellElement.addClass(rowCustomizationObj.style);
                }
            }
        }

    },

    /**
     *  Retorna se a clula informada pode receber o estilo customizad no momento.
     *  Ela no poder receber caso esteja com foco ou esteja na linha atualmente selecionada.
     * @param cellElement o elemento que representa a clula
     * @returns {Boolean} true se  permitido aplicar o estilo customizado na clula, do contrrio false.
     */
    canApplyCustomStyle : function(cellElement) {
        var rowElement = Ext.get(cellElement.parent('div', 'x-grid3-row'));
        return !(cellElement.hasClass(this.view.appliedSelectedCellClass) || rowElement.hasClass(this.view.appliedSelectedRowClass[0]) || rowElement.hasClass(this.view.appliedFocusedRowClass));
    },

    /**
     * Aplica as customizaes. O objeto que representa as customizaes deve seguir o mesmo padro definido para o atributo "customizations".
     * 
     * @public
     */
    applyCustomizations : function() {
        this.clearInvalidGridHints();

        var customization = this.customizations;
        /* faz a verificao para testar o novo valor do customization     */
        if (customization == undefined || customization == null) {
            return;
        }

        if (!customization.colCustomizations && !customization.rowCustomizations) {
            return;
        }

        var cm = this.getColumnModel();
        var colCustomizations = customization.colCustomizations;
        var rowCustomizations = customization.rowCustomizations;

        /* Itera pelas linhas e aplica os estilos nas que possuem. */
        for ( var rowIndex = 0, rowLen = this.getStore().getCount(); rowIndex < rowLen; rowIndex++) {

            var record = this.getStore().getAt(rowIndex);

            /*verifica se h customizao para o registro*/
            var rowCustomization = rowCustomizations ? rowCustomizations[record.id] : undefined;
            var rowCustomizationObj = null;
            var cellCustomizations = null;
            if (rowCustomization) {
                cellCustomizations = rowCustomization.cellCustomizations;
                rowCustomizationObj = rowCustomization.rowCustomization;
            }

            /* Itera pelas colunas. */
            for ( var colIndex = 0, colLen = cm.getColumnCount(); colIndex < colLen; colIndex++) {
                var fieldName = cm.getDataIndex(colIndex);
                var cellElement = fieldName != '0' ? this.getCellByRecord(record, fieldName) : this.getCellByIndex(rowIndex, colIndex);

                this.applyCustomStyleClass(cellElement, cellCustomizations, colCustomizations, rowCustomizationObj, fieldName);

                /*
                 * Aplica os hints, se houver.
                 * A preferencia  das celula sobre as colunas, e das colunas sobre as linhas
                 */

                var cellCustomizationObj = cellCustomizations ? cellCustomizations[fieldName] : undefined;
                var colCustomizationObj = colCustomizations ? colCustomizations[fieldName] : undefined;

                if (cellCustomizationObj && cellCustomizationObj.hints) {
                    this.applyGridHints(cellElement, cellCustomizationObj.hints, true);
                } else if (colCustomizationObj && colCustomizationObj.hints) {
                    this.applyGridHints(cellElement, colCustomizationObj.hints);
                } else if (rowCustomizationObj && rowCustomizationObj.hints) {
                    this.applyGridHints(cellElement, rowCustomizationObj.hints);
                }
            }
        }
    },

    /**
     * Aplica os hints a um determinado elemento da grid, pode ser uma linha ou uma clula.
     * 
     * @private
     * @param element HTMLElement ou Ext.Element que receber o hint
     * @param hints lista de hints
     * @param cellHint   um booleano que indica se o hint deve ser exibido para a clula
     */
    applyGridHints : function(element, hints, cellHint) {
        var tooltip = this.getView().createHint(element, hints, cellHint);
        if (tooltip) {
            if (this.gridHints == null) {
                this.gridHints = new Array();
            }
            this.gridHints[this.gridHints.length] = tooltip;
        }
    },

    /**
     * Remove o tooltip do elemento passado por parmetro.
     * 
     * @private
     * @param element  elemento que possui o hint que ser destrudo
     */
    removeGridHints : function(element) {
        var hints = this.getGridHints(element);
        if (hints && hints.length > 0) {
            for ( var i = 0; i < this.gridHints.length; i++) {
                this.gridHints.splice(i, 1);
            }
            Ext.destroy(hints);
        }
    },

    /**
     * Retorna os hints de um determinado elemento da grid.
     * 
     * @private
     * @param element elemento que possui o hint que ser destrudo
     * @return hint do elemento
     */
    getGridHints : function(element) {
        if (this.gridHints) {
            for ( var i = 0; i < this.gridHints.length; i++) {
                if (this.gridHints[i].target.id == element.id) {
                    return this.gridHints[i];
                }
            }
        }
    },

    /**
     * Quando ocorre um refresh na grid, os elementos (clulas) so recriadas e acabam tento seu id diferente do target do hint.
     * Este metodo server para remover os hints invalidos.
     * @private
     */
    clearInvalidGridHints : function() {
        if (this.gridHints) {
            Ext.destroy(this.gridHints);
            for ( var i = 0; i < this.gridHints.length; i++) {
                var hintTargetElement = this.gridHints[i].target ? Ext.get(this.gridHints[i].target.id) : null;
                if (!hintTargetElement) {
                    this.gridHints[i].destroy();
                    this.gridHints.splice(i, 1);
                    i--;
                }
            }
        }
    },

    /**
     * Retorna uma instancia de Ext.Element correspondente a clula da grid.
     * 
     * @public
     * @param field nome do campo da coluna da clula
     * @param row   indice da linha da clula
     * @return Ext.Element clula da grid
     */
    getCell : function(field, row) {
        var col = this.getColumnModel().findColumnIndex(field);
        return this.getCellByIndex(row, col);
    },

    /**
     * Retorna uma instancia de Ext.Element correspondente a clula da grid.
     * 
     * @public
     * @param row indice da linha da clula
     * @param col indice da coluna da clula
     * @return Ext.Element clula da grid
     */
    getCellByIndex : function(row, col) {
        var htmlCell = this.getView().getCell(row, col);
        return Ext.get(htmlCell);
    },

    /**
     * Retorna uma instancia de Ext.Element correspondente a clula da grid.
     * 
     * @private
     * @param record registro da grid
     * @param fieldName nome do campo da coluna da clula
     * @return Ext.Element clula da grid
     */
    getCellByRecord : function(record, fieldName) {
        var row = this.getStore().indexOfId(record.id);
        return this.getCell(fieldName, row);
    },

    /**
     * Retorna uma instancia de Ext.Element correspondente a uma linha da grid.
     * 
     * @public
     * @param row
     *            ndice da linha
     * @return Ext.Element linha da grid
     */
    getRow : function(row) {
        return Ext.get(this.getView().getRow(row));
    },

    /**
     * Define qual o estilo de highlight de registro focado na grid. CELL para exibir apenas a clula focada ROW para exibir alm da clula, a linha focada NONE para no exibir nem a linha nem a
     * clula focada.
     * 
     * @param highligh estilo de highlight (CELL, ROW, NONE)
     * 
     * @public
     */
    setHighlightFocused : function(highlight) {
        this.getView().setHighlightFocused(highlight);
        if (this.getSelectionModel().hasSelection()) {
            this.getView().updateHighlightFocused();
        }
    },

    /*
     * Funes para alterao dos atributos das colunas dinamicamente.
     */

    /**
     * @public
     */
    setColumnEditability : function(columnId, editable) {
        var colConfig = this.getColumnConfigById(columnId);
        colConfig.editable = editable;
    },

    /**
     * @public
     */
    setColumnAlignment : function(columnId, align) {
        var colConfig = this.getColumnConfigById(columnId);
        colConfig.align = align;
    },

    /**
     * @public
     */
    setColumnHeader : function(columnId, header) {
        var colConfig = this.getColumnConfigById(columnId);
        colConfig.header = header;
    },

    /**
     * @public
     */
    setColumnWidth : function(columnId, width, updateView) {
        var cm = this.getColumnModel();
        var colIndex = this.getColumnIndexById(columnId);

        cm.setColumnWidth(colIndex, width, !updateView);
    },

    updateMixedComponents : function(columnId, addComponents, removeComponents) {
        var cm = this.getColumnModel();
        var column = cm.getColumnById(columnId);

        column.removeMixedComponents(eval('(' + removeComponents + ')'));
        column.addMixedComponents(eval('(' + addComponents + ')'));
    },

    /**
     * @public
     */
    setAllowDelete : function(allowDelete) {
        this.allowDelete = allowDelete;
    },

    /**
     * @public
     */
    isAllowDelete : function() {
        return this.allowDelete;
    },

    /**
     * @public
     */
    setAllowInsert : function(allowInsert) {
        this.allowInsert = allowInsert;
    },

    /**
     * @public
     */
    isAllowInsert : function() {
        return this.allowInsert;
    },

    /**
     * @public
     */
    isReadOnly : function() {
        return this.readOnly;
    },

    /**
     * @public
     */
    setColumnVisibility : function(columnId, visible, updateView) {
        var cm = this.getColumnModel();

        var colIndex = this.getColumnIndexById(columnId);

        /* 
         * Se deve atualizar a view diretamente, chama o mtodo do ColumnModel. Se no apenas altera a configurao
         * da coluna, fazendo com que ela tenha a visibilidade alterada no prximo refresh. 
         */
        var hidden = !visible;
        if (updateView) {
            cm.setHidden(colIndex, hidden, updateView);
        } else {
            var colConfig = cm.getColumnAt(colIndex);
            colConfig.hidden = hidden;
        }
    },

    /**
     * Redefine as colunas da grid. O parmetro <code>columns</code> pode conter tanto ids de colunas j existentes como configuraes de novas colunas.<br>
     * A ordem das colunas ser de acordo com a ordem dos <code>columns</code>.
     * 
     * @public
     */
    configureColumns : function() {
        var columns = arguments;
        var cm = this.getColumnModel();

        /* Cria o array das novas colunas. */
        var newColumns = [];
        for ( var i = 0; i < columns.length; i++) {
            var column = columns[i];

            /* Se for string, refere-se ao id de uma coluna j existente. */
            if (typeof column == 'string') {
                var colIndex = this.getColumnIndexById(column);
                var colConfig = cm.getColumnAt(colIndex);
                newColumns.push(colConfig);

                /* 
                 * Remove a coluna do config atual do ColumnModel para evitar que ela tenha o editor removido pela
                 * limpeza que  feita antes da reconfigurao das colunas (na funo ColumnModel.setConfig).
                 */
                cm.config.splice(colIndex, 1);
            } else {
                /* Se no for uma string, tem que ser a configurao de uma nova coluna. */
                newColumns.push(column);
            }
        }

        /* Configura as colunas no column model */
        cm.setConfig(newColumns);
        this.applyCustomizations();
        this.getView().fireEvent('viewupdated', this.getView());
    },

    /**
     * Obtm a configurao de uma coluna atravs de seu id. Se o id no for referente a nenhuma coluna, uma exceo ser lanada.
     * 
     * @private
     */
    getColumnConfigById : function(columnId) {
        var colConfig = this.getColumnModel().getColumnById(columnId);
        if (!colConfig) {
            throw "Inexistent column: " + columnId;
        }

        return colConfig;
    },

    /**
     * Obtm o ndice de uma coluna atravs de seu id. Se o id no for referente a nenhuma coluna, uma exceo ser lanada.
     * 
     * @private
     */
    getColumnIndexById : function(columnId) {
        var index = this.getColumnModel().getIndexById(columnId);
        if (index < 0) {
            throw "Inexistent column: " + columnId;
        }

        return index;
    },

    /**
     * 
     * @param r
     * @param field
     * @returns
     * @overide
     */
    preEditValue : function(r, field) {
        var value;
        if (r.fields.get(field + this.RAW_VALUE_SUFFIX) != null) {
            value = r.data[field + this.RAW_VALUE_SUFFIX];
        } else {
            value = r.data[field];
        }

        return this.autoEncode && Ext.isString(value) ? Ext.util.Format.htmlDecode(value) : value;
    },

    /**
     * Mtodo sobrescrito para resolver as tarefas x e y.
     * A atualizao do doritos e do estilo de erro do activeEditor era feita apenas no response do servidor.
     * Isto fazia com que o doritos e o estilo piscassem cada vez que o usurio mudasse a seleo de um
     * registro com alterao/erro para um sem e vice-versa.
     * 
     * @param row linha selecionada
     * @param col coluna selecionada
     */
    startEditing : function(row, col) {
        Ext.ux.senior.grid.EditorGridPanel.superclass.startEditing.call(this, row, col);

        if (this.activeEditor) {
            var field = this.colModel.getDataIndex(col);

            this.activeEditor.field.markChanged(this.store.getAt(row).modified && this.store.getAt(row).modified[field]);

            if (this.store.getAt(row).errorFields.indexOf(field) != -1) {
                this.activeEditor.field.setHasError(true);
                this.activeEditor.field.addClass("component-error-editbox-border-size");
                this.activeEditor.field.addClass("invalid-field");
                this.activeEditor.field.addClass("component-error-editbox-border-color");
            } else {
                this.activeEditor.field.setHasError(false);
                this.activeEditor.field.setStyle("[]");
            }
            
            this.activeEditor.getEl().setStyle("overflow", "hidden");
        }
    }
});

Ext.reg('senioreditorgrid', Ext.ux.senior.grid.EditorGridPanel);