/*!
 * Ext JS Library 3.2.1
 * Copyright(c) 2006-2010 Ext JS, Inc.
 * licensing@extjs.com
 * http://www.extjs.com/license
 */
/**
 * @class Ext.ux.senior.grid.CheckboxSelectionModel
 * @extends Ext.ux.senior.grid.MultiSelectionModel A custom selection model that renders a column of checkboxes that can be toggled to select or
 *          deselect rows.
 * @constructor
 * @param {Object}
 *            config The configuration options
 */
Ext.ux.senior.grid.CheckboxSelectionModel = Ext.extend(Ext.ux.senior.grid.MultiSelectionModel, {

    /**
     * @cfg {Boolean} checkOnly <tt>true</tt> if rows can only be selected by clicking on the checkbox column (defaults to <tt>false</tt>).
     */
    /**
     * @cfg {String} header Any valid text or HTML fragment to display in the header cell for the checkbox column. Defaults to:
     * 
     * <pre><code>
     * '&lt;div class=&quot;x-grid3-hd-checker&quot;&gt; &amp;#160;&lt;/div&gt;'
     * </tt>
     * </code></pre>
     * 
     * The default CSS class of <tt>'x-grid3-hd-checker'</tt> displays a checkbox in the header and provides support for automatic check
     *      all/none behavior on header click. This string can be replaced by any valid HTML fragment, including a simple text string (e.g., <tt>'Select
     *      Rows'</tt>), but the automatic check all/none behavior will only work if the <tt>'x-grid3-hd-checker'</tt> class is supplied.
     */
    header : '<div class="x-grid3-hd-checker">&#160;</div>',
    /**
     * @cfg {Number} width The default width in pixels of the checkbox column (defaults to <tt>20</tt>).
     */
    width : 20,
    /**
     * @cfg {Boolean} sortable <tt>true</tt> if the checkbox column is sortable (defaults to <tt>false</tt>).
     */
    sortable : false,

    // private
    menuDisabled : true,
    fixed : true,
    hideable : false,
    dataIndex : '0',
    id : 'checker',

    constructor : function() {
        Ext.ux.senior.grid.CheckboxSelectionModel.superclass.constructor.apply(this, arguments);

        if (this.checkOnly) {
            this.handleMouseDown = Ext.emptyFn;
        }
    },

    // private
    initEvents : function() {
        Ext.ux.senior.grid.CheckboxSelectionModel.superclass.initEvents.call(this);
        this.grid.on('render', function() {
            var view = this.grid.getView();
            view.mainBody.on('mousedown', this.onMouseDown, this);
            Ext.fly(view.innerHd).on('mousedown', this.onHdMouseDown, this);

            if (view.lockedBody && view.lockedHd) {
                view.lockedBody.on('mousedown', this.onMouseDown, this);
                Ext.fly(view.lockedHd).on('mousedown', this.onHdMouseDown, this);
            }

        }, this);
    },

    // If handleMouseDown was called from another event (enableDragDrop), set a flag so
    // onMouseDown does not process it a second time
    handleMouseDown : function() {
        this.mouseHandled = true;
    },

    // private
    onMouseDown : function(e, t) {
        if (e.button === 0 && t.className == 'x-grid3-row-checker') { // Only fire if left-click
            e.stopEvent();
        }
        this.mouseHandled = false;
    },

    // private
    onHdMouseDown : function(e, t) {
        if (t.className == 'x-grid3-hd-checker') {
            e.stopEvent();
            var hd = Ext.fly(t.parentNode);
            var isChecked = hd.hasClass('x-grid3-hd-checker-on');
            if (isChecked) {
                hd.removeClass('x-grid3-hd-checker-on');
                this.grid.getColumnConfigById(this.id).attr= '';
                this.select0("CLEAR");
            } else {
                hd.addClass('x-grid3-hd-checker-on');
                this.grid.getColumnConfigById(this.id).attr = 'class=x-grid3-hd-checker-on';
                this.select0("ALL");
            }
        }
    },

    // private
    renderer : function(v, p, record) {
        return '<div class="x-grid3-row-checker">&#160;</div>';
    },

    getHeaderTooltip : function() {
        return null;
    },

    // private
    select0 : function(selection) {
        if (this.locked) {
            return;
        }
        this.clearSelections();
        this.grid.fireEvent("selection", this.grid, "selectRow", "ROW", selection, -1);
    },

    /**
     * Dispara o evento de click de uma linha da grid.
     * 
     * @param grid
     *            componente que sofreu o click
     * @param row
     *            indice da linha que sofreu o click
     * @param event
     *            evento
     * @private
     */
    rowClickHandler : function(grid, row, event) {
        if (row < 0 || row >= this.grid.store.getCount()) {
            return;
        }
        if (event.target.className == "x-grid3-row-checker") {// se a linha sofreu click no checkbox
            event.ctrlKey = true;
        }
        this.fireSelectionEvent("selectRow", "ROW", event, row);
    }
});