/**
 * Classe que controla seleo simples de clula em uma Grid. Caso uma coluna necessite controlar se suas clulas podem ou no ser selecionadas deve implementar o mtodo
 * <code>boolean isSelectable()</code>
 * 
 * @author Patrick.Nascimento
 */
Ext.ux.senior.grid.CellSelectionModel = Ext.extend(Ext.grid.CellSelectionModel, {

    /* index da ltima coluna selecionada. Usado para quando os dados so atualizados e seleo  perdida */
    lastCol : 0,

    /*  ltima chave de registro selecionada */
    lastRecord : null,

    /**
     * @cfg {Number} initialSelectedRow Linha que deve aparecer selecionada inicialmente.
     */

    /**
     * Sobrescrever o init para inicializar o lastCol como em uma clula que seja selecionvel.
     * 
     * @private
     */
    init : function(grid) {
        Ext.ux.senior.grid.CellSelectionModel.superclass.init.call(this, grid);
        for ( var col = 0; col < grid.getColumnModel().getColumnCount(false); col++) {
            if (this.isSelectable(0, col, grid.getColumnModel())) {
                return;
            }
            this.lastCol++;
        }
    },

    /**
     * @override
     */
    initEvents : function() {
        Ext.ux.senior.grid.CellSelectionModel.superclass.initEvents.call(this);
        this.grid.on('viewready', this.onViewReady, this, {
            single : true
        });
        this.grid.on('cellclick', this.handleClick, this);
    },

    handleTab : function(shift) {
        var selectedRow = null;
        var selectedColumn = null;
        var selection = this.getSelectedCell();

        if (selection) {
            selectedRow = selection[0];
            selectedColumn = selection[1];
        } else {
            selectedColumn = this.grid.lastCol;
        }

        if (this.canNavigate()) {
            var store = this.grid.getStore();
            if (selectedRow == store.getCount() - 1 && store.isAtEof()) {

                var record = this.grid.getStore().getAt(selectedRow);

                var isFirstColumn = selectedColumn == this.getFirstSelectableColumn(selectedRow);
                var isLastColumn = selectedColumn == this.getLastSelectableColumn(selectedRow);

                /* Se estiver na ltima coluna do ltimo registro, este registro estiver sujo e
                 * o shift no estiver pressionado, tenta inserir um novo registro.
                 * Caso for a primeira coluna, o registro no estiver sujo e o shift estiver pressionado,
                 * tenta excluir este registro.
                 */
                if (!shift) {
                    if (isLastColumn && (!record.phantom || (record.phantom && record.dirty))) {
                        /* Ajusta o lastCol para a posio que a coluna deve assumir caso o registro de edio rpida for inserido.
                         * Se ele no for inserido, o lastCol definido ser irrelevante. 
                         */
                        this.lastCol = this.getFirstSelectableColumn(selectedRow);
                        
                        if(this.grid.isAllowInsert()) {
                            this.grid.fireEvent('tryinsertfasteditrecord');
                        } else {
                            //Cancela o evento da tecla TAB e executa o foco no prximo componente, o evento  cancelado para no acontecer um "TAB extra" no componente focado
                            Ext.EventObject.stopEvent();
                            FocusManager.focusNextComponent();
                        }
                    } else if(isLastColumn) {
                        //Cancela o evento da tecla TAB e executa o foco no prximo componente, o evento  cancelado para no acontecer um "TAB extra" no componente focado
                        Ext.EventObject.stopEvent();
                        FocusManager.focusNextComponent();
                    }
                } else if (isFirstColumn && record.phantom && !record.dirty) {
                    /* Ajusta o lastCol para a posio que a coluna deve assumir caso o registro de edio rpida for excludo.
                     * Se ele no for excludo, o lastCol definido ser irrelevante.
                     */
                    if(this.grid.isAllowInsert()) {
                        this.lastCol = this.getLastSelectableColumn(selectedRow);
                        this.grid.fireEvent('trydeletefasteditrecord');
                    } else {
                        //Cancela o evento da tecla TAB e executa o foco no componente anterior, o evento  cancelado para no acontecer um "TAB extra" no componente focado
                        Ext.EventObject.stopEvent();
                        FocusManager.focusPrevComponent();
                    }
                }
            } else if(selectedRow == 0){
                var isFirstColumn = selectedColumn == this.getFirstSelectableColumn(selectedRow);
                if(shift && isFirstColumn) {
                    //Cancela o evento da tecla TAB e executa o foco no componente anterior, o evento  cancelado para no acontecer um "TAB extra" no componente focado
                    Ext.EventObject.stopEvent();
                    FocusManager.focusPrevComponent();
                }
            }
        } else {
            if (!shift) {
                //Cancela o evento da tecla TAB e executa o foco no prximo componente, o evento  cancelado para no acontecer um "TAB extra" no componente focado
                Ext.EventObject.stopEvent();
                FocusManager.focusNextComponent();
            } else {
                //Cancela o evento da tecla TAB e executa o foco no componente anterior, o evento  cancelado para no acontecer um "TAB extra" no componente focado
                Ext.EventObject.stopEvent();
                FocusManager.focusPrevComponent();
            }
        }
    },

    /**
     * Executa as aes que s podem ser feitas depois que a view da grid j foi renderizada.
     * 
     * @private
     */
    onViewReady : function() {
        /* Seleciona as linhas iniciais. */
        if (this.initialSelectedRow !== undefined) {
            this.select(this.initialSelectedRow);

            delete this.initialSelectedRow;
        }
    },

    fireScroll : function() {
        if (this.reqObj) {
            var view = this.grid.getView();
            view.fireScrollEvent(this.reqObj.type, this.reqObj.row, true);
            this.reqObj = null;
        }
    },

    handleClick : function() {
        this.fireScroll();
    },

    listeners : {
        "beforecellselect" : function(sm, row, col) {

            var selection = sm.getSelectedCell();
            if (row < 0 || row > this.grid.getStore().getCount() - 1) {
                selection = null;
            }

            if (col !== undefined && this.isSelectable(row, col, this.grid.getColumnModel())) {
                /* guarda a coluna selecionada. */
                this.lastCol = col;
            }
            /* se a coluna no puder receber foco j retorna false aqui */
            if (this.grid.getColumnModel().getColumnAt(col).isSelectable) {
                if (!this.grid.getColumnModel().getColumnAt(col).isSelectable()) {
                    return false;
                }
            }

            /* se a seleo veio do server, atualiza o ultimo registro selecionado tambm */
            if (this.server === true) {
                this.reqObj = null;
                this.lastRecord = this.grid.getStore().getAt(row);
                this.server = false;
            } else {
                if (!selection) {
                    var key = Ext.EventObject.getKey();
                    switch (key) {
                    case Ext.EventObject.UP:
                        this.grid.view.fireScrollEvent(ScrollEvent.UP, -1, true);
                        return false;
                    case Ext.EventObject.DOWN:
                        this.grid.view.fireScrollEvent(ScrollEvent.DOWN, -1, true);
                        return false;
                    case Ext.EventObject.PAGEDOWN:
                        if (!Ext.EventObject.ctrlKey) {
                            this.grid.view.fireScrollEvent(ScrollEvent.PAGEDOWN, -1, true);
                            return false;
                        }
                        break;
                    case Ext.EventObject.PAGEUP:
                        if (!Ext.EventObject.ctrlKey) {
                            this.grid.view.fireScrollEvent(ScrollEvent.PAGEUP, -1, true);
                            return false;
                        }
                        break;
                    }
                    selection = [ -1, -1 ];
                }

                var view = sm.grid.getView();
                var type;
                if (row != selection[0]) {
                    this.lastRecord = this.grid.getStore().getAt(row);
                    var key = Ext.EventObject.getKey();
                    switch (key) {
                    case Ext.EventObject.UP:
                        type = ScrollEvent.UP;
                        break;
                    case Ext.EventObject.DOWN:
                        type = ScrollEvent.DOWN;
                        break;
                    case Ext.EventObject.PAGEDOWN:
                        type = ScrollEvent.PAGEDOWN;
                        break;
                    case Ext.EventObject.PAGEUP:
                        type = ScrollEvent.PAGEUP;
                        break;
                    case Ext.EventObject.HOME:
                        type = ScrollEvent.FIRST;
                        break;
                    case Ext.EventObject.END:
                        type = ScrollEvent.LAST;
                        break;
                    default:
                        type = ScrollEvent.SELECT;
                    }
                    if (Ext.EventObject.isNavKeyPress()) {
                        view.fireScrollEvent(type, row, true);
                    } else {
                        this.reqObj = {
                            type : type,
                            row : row
                        };
                    }
                }
            }
            return true;

        }
    },

    /**
     *  
     * @override
     */
    handleMouseDown : function(g, row, cell, e) {
        //ignora o mousedown quando est editando para no focar a linha, apenas excluir o registro
        if (this.grid.getColumnModel().getColumnAt(cell).xtype != "deletecolumn") {
            Ext.ux.senior.grid.CellSelectionModel.superclass.handleMouseDown.call(this, g, row, cell, e);
        }
    },

    /**
     * Mtodo que auxilia o servidor na seleo de uma clula.
     * 
     * @param row linha a ser selecionada
     * @param serverForce define se deve forar a seleo com as informaes do servidor. <code>true</code> para forar a seleo do servidor. <code>false</code> o cliente define qual a clula que deve ser selecionada.
     */
    serverSelect : function(row, serverForce) {
        if (this.lastRecord != null) {
            if (!serverForce) {
                var lastRow = this.grid.getStore().indexOfId(this.lastRecord.id);
                row = lastRow > -1 ? lastRow : row;
            }
        }
        this.server = true;
        this.select(row, this.lastCol, null, true);
    },

    /**
     * 
     * @override
     */
    select : function(row, col, prevView, prevFocus, rec) {
        if (col === null || col === undefined) {
            col = this.lastCol;

            var view = this.grid.getView();
            var isVisible = row >= view.getFirstRecordIndex() - 1 && row <= (view.getFirstRecordIndex() + view.visibleRows + 1);
            if (!isVisible || this.grid.isEditing) {
                /* focar no body */
                prevFocus = true;
            }

        } else if (this.isSelectable(row, col, this.grid.getColumnModel())) {
            if (!this.server) {
                this.lastCol = col;
            }

        }

        if (!this.isSelectable(row, this.lastCol, this.grid.getColumnModel())) {
            this.lastCol = this.getFirstSelectableColumn(row);
        }
        
        Ext.ux.senior.grid.CellSelectionModel.superclass.select.call(this, row, this.lastCol, prevView, prevFocus, rec);
    },

    /**
     * Mtodo que informa se determinada clula  "selecionvel" com base no ColumnModel.  sobrescrito para tratar colunas do Framework que no podem receber foco, recurso no existente no Ext.
     * 
     * @override
     */
    isSelectable : function(rowIndex, colIndex, cm) {
        var selectable = Ext.ux.senior.grid.CellSelectionModel.superclass.isSelectable.call(this, rowIndex, colIndex, cm);
        var column = cm.getColumnAt(colIndex);
        if (column.isSelectable) {
            selectable = selectable && column.isSelectable();
        }
        return selectable;
    },

    /**
     * Mtodo que determinada se a clula  editvel e pode receber foco como editorAtivo. Sobreescrito o mtodo do ext pois ele no preve o comportamento de campos chaves que no podem ser editados
     * depois de serem salvos
     * 
     * @override
     */
    acceptsNav : function(rowIndex, colIndex, cm) {
        var acceptsNav = Ext.ux.senior.grid.CellSelectionModel.superclass.acceptsNav.call(this, rowIndex, colIndex, cm);
        var record = this.grid.view.ds.getAt(rowIndex);
        return acceptsNav && this.grid.isColumnEditable(record, colIndex);
    },

    /**
     * Trata eventos de keydown do GridPanel.
     * 
     * @param {Number}
     *            key keyCode da tecla pressionada.
     * @param {Boolean}
     *            ctrl <code>true</code> caso a tecla ctrl esteja pressionada.
     * @param {Boolean}
     *            shift <code>true</code> caso a tecla shift esteja pressionada.
     *  
     * @protected
     */
    onKeyDown : function(event, key, ctrl, shift) {
        var E = Ext.EventObject;
        var grid = this.grid, view = grid.getView();

        if (key === E.INSERT) {
            if (ctrl && !this.grid.isReadOnly()) {
                this.grid.fireEvent("insertRow");
            }
        }

        if (grid.store.getCount() == 0) {
            return;
        }

        /*
         * Se houver clula selecionada no client, o "selectedRow" ir apontar para a linha selecionada e o "selectedColumn" para a coluna selecionada.
         * Se no houver clula selecionada no client, o "selectedRow" ser undefined e o "selectedColumn" ir apontar para a ltima coluna selecionada.
         * 
         * Ou seja, o "selectedRow" s poder ser utilizado se "this.canNavigate()" == true.
         */
        var selection = this.getSelectedCell(); /* [row,col] */
        var selectedRow = null;
        var selectedColumn = null;
        if (selection) {
            selectedRow = selection[0];
            selectedColumn = selection[1];
        } else {
            selectedColumn = this.lastCol;
        }

        var firstVisibleRow = view.getFirstRecordIndex();
        var lastVisibleRow = firstVisibleRow + view.visibleRows - 1;
        switch (key) {
        case E.PAGE_UP:
            if (ctrl) {
                /* Foca o primeiro registro da pgina. */
                view.stopEvents();
                this.selectFirstRecord(grid);
            } else {
                /*  se for autoheight no faz nada, s tem uma pgina */
                if (grid.autoHeight) {
                    return;
                }
                /* Navega para a pgina anterior. */
                var rowToSelect = -1;
                if (this.canNavigate()) {
                    if (selectedRow >= firstVisibleRow && selectedRow <= lastVisibleRow) {
                        rowToSelect = firstVisibleRow - view.visibleRows;
                    } else {
                        rowToSelect = selectedRow - view.visibleRows;
                    }

                }
                if (grid.getStore().isAtBof() && rowToSelect < 0) {
                    this.select(0, selectedColumn);
                } else {
                    this.select(rowToSelect, selectedColumn);
                    /* Rolou um pageUp, a lgica do pageUp diz que  necessrio subir uma pgina e focar o primeiro registro da pgina.
                     * Para subir uma pgina basta mostrar como primeiro registro visvel a linha que tem o ndice igual  "rowToSelect"*/
                    view.stopEvents();
                    view.showRecordAtFirst(rowToSelect);
                    view.resumeEvents();
                }
            }
            Ext.EventObject.stopEvent();
            return false;
        case E.PAGE_DOWN:
            if (ctrl) {
                /* Foca o ltimo registro da pgina. */
                view.stopEvents();
                this.selectLastRecord(grid);
            } else {
                /*  se for autoheight no faz nada, s tem uma pgina */
                if (grid.autoHeight) {
                    return;
                }
                /* Navega para a prxima pgina. */
                /*A partir deste ndice ser feito o pageDown, ou seja o registro abaixo deste ndice ser o primeiro visvel da nova pgina*/
                var indexPageDownRow = -1;
                var rowToSelect = -1;
                if (this.canNavigate()) {
                    if (selectedRow >= firstVisibleRow && selectedRow <= lastVisibleRow) {
                        indexPageDownRow = lastVisibleRow;
                    } else {
                        indexPageDownRow = selectedRow;
                    }
                    rowToSelect = indexPageDownRow + view.visibleRows;
                }
                if (grid.getStore().isAtEof() && rowToSelect > grid.getStore().getCount() - 1) {
                    this.select(grid.getStore().getCount() - 1, selectedColumn);
                } else {
                    this.select(rowToSelect, selectedColumn);
                    if (indexPageDownRow > -1) {
                        /* Rolou um pageDown, a lgica do pageDown diz que  necessrio descer uma pgina e focar o ltimo registro
                         * para descer uma pgina basta mostrar como primeiro registro visvel a linha aps o "indexPageDownRow"*/
                        view.stopEvents();
                        view.showRecordAtFirst(indexPageDownRow + 1);
                        view.resumeEvents();
                    }
                }
            }
            Ext.EventObject.stopEvent();
            return false;

        case E.HOME:
            if (ctrl) {
                if (grid.getStore().isAtBof()) {
                    this.select(0, selectedColumn);
                } else {
                    /* Navega e foca o primeiro registro. */
                    view.beforeScroll();
                    view.scrollFirst(true);
                    view.afterScroll(ScrollEvent.FIRST);
                }
            } else {
                if (this.canNavigate()) {
                    /* Navega para a primeira coluna que pode receber foco. */
                    this.select(selectedRow, this.getFirstSelectableColumn(selectedRow));
                }
            }
            break;

        case E.END:
            if (ctrl) {
                if (grid.getStore().isAtEof()) {
                    this.select(grid.getStore().getCount() - 1, selectedColumn);
                } else {
                    /* Navega e foca o ltimo registro. */
                    view.beforeScroll();
                    view.scrollLast(true);
                    view.afterScroll(ScrollEvent.LAST);
                    Ext.EventObject.keyCode = 0;
                }
            } else {
                if (this.canNavigate()) {
                    /* Navega para a ltima coluna. */
                    this.select(selectedRow, this.grid.getColumnModel().getColumnCount() - 1);
                }
            }
            break;

        case E.F2:
            if (this.canNavigate()) {
                /* inicia o editor na clula que est selecionada. */
                this.grid.startEditing(selectedRow, selectedColumn);
            }
            break;
        case E.UP:
            if (this.canNavigate()) {
                if (selectedRow == this.grid.getStore().getCount() - 1) {
                    /* S permite o envio do evento de tryDeleteFastEditRecord se a seleo estiver na ltima 
                     * linha da grid e a ultima for uma linha nova.
                     */
                    var record = this.grid.getStore().getAt(selectedRow);
                    if (this.grid.getStore().isAtEof() && record.phantom && !record.dirty) {
                        this.grid.fireEvent('trydeletefasteditrecord');
                        return false;
                    }
                } else if (selectedRow == 0) {
                    this.grid.view.stopEvents();
                    this.grid.view.showRecordAtFirst(0);
                    this.grid.view.resumeEvents();
                }
            }
            break;
        case E.DOWN:
            if (this.canNavigate()) {
                if (selectedRow == this.grid.getStore().getCount() - 1) {
                    /* S permite o envio do evento de tryInsertFastEditRecord se a seleo estiver na ltima 
                     * linha da grid e a ltima no for uma linha nova. 
                     */
                    var record = this.grid.getStore().getAt(selectedRow);
                    if (this.grid.getStore().isAtEof() && (!record.phantom || (record.phantom && record.dirty))) {
                        this.grid.fireEvent('tryinsertfasteditrecord');
                        return false;
                    }

                    this.grid.view.stopEvents();
                    this.grid.view.showRecordAtFirst(this.getSelectedCell()[0]);
                    this.grid.view.resumeEvents();
                }
            }

            break;
        case E.ESC:
            if (this.canNavigate()) {
                /* Verifica se o registro focado  "inserido". */
                var focusedRecord = this.grid.getStore().getAt(selectedRow);
                if (focusedRecord.phantom) {
                    /* TODO verificar sobre a validao de readOnly!  ruim cada lugar que lana o "deleteRow" ter que verificar isto. */
                    var record = this.grid.getStore().getAt(selectedRow);
                    if (!this.grid.readOnly && !this.grid.editing && !record.dirty) {
                        this.grid.fireEvent("deleteRow", selectedRow);
                    }
                }
            }
            break;
        case E.TAB:
            if (this.canNavigate()) {
                this.handleTab(shift);
            }
        }

        return true;
    },

    /**
     * Retorna se  possvel navegar pelas clulas da grid.
     * 
     * @private
     * @return {boolean} true se existe uma clula selecionada, false se no existir
     */
    canNavigate : function() {
        /* Se existe uma seleo no SelectionManager existe registros na grid, e se existem registros na grid,  possvel navegar entre eles. */
        return this.getSelectedCell() !== null;
    },

    /**
     * 
     * Retorna a primeira coluna disponvel para seleo
     * 
     * @param {}
     *            selection seleo da qual se deseja obter a coluna *
     * @return {Number} primeira coluna que permite seleo na linha
     * 
     * @private
     */
    getFirstSelectableColumn : function(row) {
        var col = 0;
        while (!this.isSelectable(row, col, this.grid.getColumnModel())) {
            col++;
        }
        return col;
    },

    /** 
     * Retorna a ltima coluna disponvel para seleo
     * 
     * @param {Array} selection seleo da qual se deseja obter a coluna
     * @return {Number} primeira coluna que permite seleo na linha ou -1 se no houver coluna selecionvel.
     * 
     * @private
     */
    getLastSelectableColumn : function(row) {
        var cm = this.grid.getColumnModel();
        var columnCount = cm.getColumnCount(false);

        if (columnCount == 0) {
            throw "There is no column to select";
        }

        /* Procura pela ltima coluna selecionvel. */
        for ( var col = columnCount - 1; col >= 0 && !this.isSelectable(row, col, cm); col--) {
        }

        return col;
    },

    /**
     * Seleciona o ltimo registro visivel da grid.
     * 
     * @private
     * @param grid
     *            referencia da grid para este selection model
     */
    selectLastRecord : function(grid) {
        var view = grid.getView();
        var selectedRow = view.getFirstRecordIndex() + view.visibleRows - 1;
        /*
         * garante que a ultima linha selecionada seja uma vlida, pois pode ocorrer do store ser menor que o numero da area de linhas "visiveis" da
         * grid
         */
        var storeVisible = this.grid.getStore().getCount() - 1;
        if (selectedRow > storeVisible) {
            selectedRow = storeVisible;
        }
        this.select(selectedRow, this.lastCol);
    },

    /**
     * Selectiona o primeiro registro visivel da grid.
     * 
     * @private
     * @param grid
     *            referencia da grid para este selection model
     */
    selectFirstRecord : function(grid) {
        this.select(grid.getView().getFirstRecordIndex(), this.lastCol);
    },

    /**
     * Chama o GridView para focar um registro.<br>
     * O GridView sabe se  necessrio focar no registro ou no.<br>
     * 
     * <b> Funo chamada pelo GridPanel. </b>
     * 
     * @param {Number}
     *            index ndice do registro a ser focado.
     * @param {Boolean}
     *            forceHtmlFocus <code>true</code> para forar o GridView a setar o foco no elemento html, assim fazendo o registro ficar visvel.
     * 
     * @protected
     */
    focusRow : function(index, forceHtmlFocus) {
        var view = this.grid.getView();
        view.internalFocusRow(index, forceHtmlFocus);
    },

    /**
     * Seleciona uma clula com base no fieldName que representa as colunas.<br>
     * <br>
     * Esta seleo s  feita caso j exista outra seleo, onde apenas a coluna selecionada poder mudar, a linha continua a mesma.<br>
     * Caso no exista seleo nada acontece.
     * 
     * @param {String}
     *            fieldName o nome do field que representa a coluna da clula a ser selecionada.
     * 
     * @public
     */
    focusCellByField : function(fieldName, activeEditor) {
        if (this.hasSelection()) {
            var selection = this.getSelectedCell();
            var cm = this.grid.getColumnModel();
            var col = cm.findColumnIndex(fieldName);
            if (col > -1) {
                if (col != selection[1]) {
                    this.select(selection[0], col);
                }
                if (activeEditor) {
                    /* Verifica antes se a clula j est editando, pois se mandar editar
                     * uma clula que j est editando, ela sai do modo de edio
                     */
                    var needStartEditing = true;
                    if (this.grid.activeEditor) {
                        var activeRow = this.grid.getStore().indexOf(this.grid.activeEditor.record);
                        var activeCol = this.grid.activeEditor.col;
                        needStartEditing = activeRow != selection[0] || activeCol != col;
                    }
                    if (needStartEditing) {
                        this.grid.startEditing(selection[0], col);
                    }
                }
            }
        }
    },

    /**
     * Funo utilizada pelas rotinas de drag&drop. Est aqui apenas para no gerar erros.<br>
     * Este SelectionModel no guarda seleo, ento no  vivel implementar.
     * 
     * <b> No utilizar pois sempre retorna um valor padro</b>
     */
    isSelected : function(index) {
        return true;
    },

    /**
     * Funo utilizada pelas rotinas de drag&drop. Est aqui apenas para no gerar erros.<br>
     * Este SelectionModel no guarda seleo, ento no  vivel implementar.
     * 
     * <b> No utilizar pois sempre retorna um valor padro</b>
     */
    getSelections : function() {
        this.grid.getStore().getAt(0);
    },

    /**
     * Funo utilizada pelas rotinas de drag&drop. Est aqui apenas para no gerar erros.<br>
     * Este SelectionModel no guarda seleo, ento no  vivel implementar, alm do que o servidor no sabe quantos so os registros que esto selecionados.
     * 
     * <b> No utilizar pois sempre retorna um valor padro</b>
     */
    getCount : function() {
        return "";
    },

    /**
     * Retorna o registro da clula selecionada.
     * @returns o regristro selecionado ou null se no houver nenhum registro selecionado
     */
    getSelected : function() {
        if (this.selection) {
            return this.selection.record;
        }
        return null;
    }

});