Ext.ns('Ext.ux.senior');

/**
 * @class Ext.ux.senior.FileUploadField
 * @extends Ext.form.TextField
 * 
 * Cria um boto de upload de arquivos
 * 
 * @xtype fileuploadfield
 */
Ext.ux.senior.FileUploadField = Ext.extend(Ext.form.TextField, {

    /**
     * @cfg {Object} buttonCfg configurao de um objeto {@link Ext.ux.senior.Button} par ao boto de upload.
     */

    // private
    readOnly : true,

    /**
     * @hide
     * @method autoSize
     */
    autoSize : Ext.emptyFn,

    // private
    initComponent : function() {
        Ext.ux.senior.FileUploadField.superclass.initComponent.call(this);

        this.addEvents(
        /**
         * @event fileselected
         * Fires when the underlying file input field's value has changed from the user
         * selecting a new file from the system file selection dialog.
         * @param {Ext.ux.senior.FileUploadField} this
         * @param {String} value The file value returned by the underlying file input field
         */
        'fileselected');

        var btnCfg = Ext.applyIf(this.buttonCfg || {}, {
            text : this.buttonText
        });

        this.button = new Ext.ux.senior.Button(Ext.apply(btnCfg, {
            cls : btnCfg.cls + ' x-senior-file-button x-form-file-btn' + (btnCfg.iconCls ? ' x-btn-icon' : ''),
            fileUploadField : this,
            getHintTarget : function() {
                return this.fileUploadField.fileInput;
            }
        }));
    },

    // private
    onRender : function(ct, position) {
        this.hidden = this.buttonCfg.hidden;
        Ext.ux.senior.FileUploadField.superclass.onRender.call(this, ct, position);

        var wrapConfig = {
            cls : 'x-form-field-wrap x-form-file-wrap x-senior-file-item-selector' + (!!this.hidden ? ' x-senior-displaynone' : '')
        };

        this.wrap = this.el.wrap(wrapConfig);
        this.el.addClass('x-form-file-text');
        this.el.dom.removeAttribute('name');

        this.createFileInput();
        this.createUploadButton();

        this.el.hide();

        this.bindListeners();
        this.resizeEl = this.positionEl = this.wrap;
    },
    
    adjustDimensions : function() {
        this.configureFileInput();
        this.adjustWrapDimensions();
    },
    
    adjustWrapDimensions : function() {
        var buttonElWidth = this.button.getEl().getWidth();
        var wrapWidth = buttonElWidth || this.button.width;
        if (wrapWidth) {
            wrapWidth += this.wrap.getBorderWidth("lr") + this.wrap.getPadding("lr");
            this.wrap.setWidth(wrapWidth);
        }
        this.wrap.setHeight(this.button.height);
    },

    doUpload : function() {
        var isUploadComplete = false;

        var listener1 = new ConnectionMessageListener();
        listener1.fileComponent = this.fileComponent;
        listener1.fileField = this;

        listener1.canSendRequest = function() {
            return true;
        };
        listener1.onComplete = function(success, fileName, responseJSON) {
            ProcessManager.finishProcess();
            isUploadComplete = true;

            if (responseJSON.errorMessage || responseJSON.errorStack) {
                this.fileComponent.fireEvent('uploadError', fileName, responseJSON.errorMessage ? responseJSON.errorMessage : "", responseJSON.errorStack);
            }

            this.fileField.fileInput.dom.value = '';
            this.fileComponent.unblock();
        };

        // ---------------------------------
        var listener2 = new ConnectionMessageListener();
        listener2.canSendRequest = function() {
            return isUploadComplete;
        };

        // ---------------------------------
        var element = document.getElementById(this.fileComponent.uploadField.getFileInputId());
        var fileName = qq.getFileName(element);

        this.fileComponent.block();

        ProcessManager.startProcess();
        this.fileComponent.fireEvent('uploadFile', listener1);
        // depois de enviar o arquivo para o servidor, mandamos uma notificao de uploadComplete para
        // o servidor. Esta notificao poderia estar na callback onComplete da mensagem de upload 
        // para o servidor mas esta aqui porque tem como objetivo trancar a fila de negcio para que
        // ela s fique disponvel aps o upload ser completo. Isto  necessrio para que fique correta
        // a ordem dos eventos que chegam ao servidor. Se aps fazer o upload, ser clicado em um button,
        // por exemplo,  necessrio que a ordem dos eventos do servidor seja: upload, button.
        this.fileComponent.fireEvent('uploadComplete', fileName, listener2); // fila de negcio

        if (Ext.isIE9 || Ext.isIE10 || Ext.isSafari) {

            /*
             * Razes para recriar o input:
             * 
             * - No ie9 o upload  realizado utilizando iframe e form, quando o form  destrudo, o input  destrudo junto.
             * - No ie10 e Safari,  recriado apenas para permitir o upload do mesmo arquivo duas vezes seguidas.
             * - No chrome no precisa recriar.
             */

            if (Ext.isIE10 || Ext.isSafari) {
                Ext.get(this.fileInput).purgeAllListeners();
                Ext.destroy(Ext.get(this.fileInput));
            }

            //recria o input
            this.createFileInput();
            this.configureFileInput();
            this.bindFileInputListeners();

            //reconfigura o hint           
            if (this.button.objTooltip) {
                this.button.objTooltip.initTarget(this.fileInput);
            }
        }
    },

    configureFileInput : function() {
        if (this.button.getHeight() == 0) {
            this.fileInput.setHeight(this.button.height);
        } else {
            this.fileInput.setHeight(this.button.getHeight());
        }

        if (this.button.getWidth() == 0) {
            this.fileInput.setWidth(this.button.width);
        } else {
            if (Ext.isIE11) {//o caret do input escondido aparece piscando por motivos de IE11
                this.fileInput.setWidth(this.button.getWidth() + 8);
            } else {
                this.fileInput.setWidth(this.button.getWidth());
            }
        }
        this.fileInput.addClass('x-btn');
    },

    bindListeners : function() {

        this.on('fileselected', function() {
            var element = document.getElementById(this.getFileInputId());

            if (Ext.isIE9) {
                // no ie9 no  possivel saber o tamanho do arquivo, ento envia direto
                this.doUpload();
            } else if (element.files) {
                // envia o tamanho antes do arquivo, se estiver ok, da o servidor ir pedir o upload do arquivo.
                //  apenas uma otimizao, aps o upload o servidor valida novamente.
                var file = element.files[0];
                var size = file.size;
                var name = file.name;

                //Quando o arquivo  vazio (size == 0) o erro  lanado pelo servidor
                if ((Ext.isNumber(size) && size >= 0) && !!name) {
                    this.fileComponent.fireEvent('fileSize', size, name);
                }

            }

        });

        this.bindFileInputListeners();
    },

    bindFileInputListeners : function() {
        this.fileInput.on({
            scope : this,
            mouseenter : function() {
                this.button.fireEvent('mouseover', this.button);
                this.button.addClass([ 'x-btn-over' ]);
            },
            mouseleave : function() {
                this.button.fireEvent('mouseout', this.button);
                this.button.removeClass([ 'x-btn-over' ]);
            },
            mousedown : function() {
                this.button.fireEvent('mousedown', this.button);
                this.button.addClass('x-btn-click', 'x-btn-focus');
            },
            mouseup : function() {
                this.button.fireEvent('mouseup', this.button);
                this.button.removeClass([ 'x-btn-focus', 'x-btn-click' ]);
            },
            focus : function() {
                this.button.addClass('x-btn-focus');
            },
            blur : function() {
                this.button.removeClass('x-btn-focus');
            },
            change : function() {
                var v = this.fileInput.dom.value;
                this.setValue(v);
                this.fireEvent('fileselected', this, v);
            },
            click : function() {
                // http://stackoverflow.com/questions/12030686/html-input-file-selection-event-not-firing-upon-selecting-the-same-file
                this.fileInput.dom.value = '';
            }
        });
    },

    createUploadButton : function() {
        this.button.render(this.wrap);
        this.button.on('disable', this.onDisable, this);
        this.button.on('enable', this.onEnable, this);
    },

    createFileInput : function() {
        var inputConfig = {
            id : this.getFileInputId(),
            name : this.name || this.getId(),
            cls : 'x-form-file',
            tag : 'input',
            type : 'file',
            size : 1
        };

        if (Ext.isWebKit || Ext.isFireFox) {
            // Para no mostrar o hint de 'Nenhum arquivo selecionado'
            inputConfig.title = "&nbsp";
        }

        this.fileInput = this.wrap.createChild(inputConfig);
    },

    reset : function() {
        if (this.rendered) {
            this.fileInput.remove();
            this.createFileInput();
            this.bindListeners();
        }
        Ext.ux.senior.FileUploadField.superclass.reset.call(this);
    },

    // private
    getFileInputId : function() {
        return this.id + '-file';
    },

    // private
    onResize : function(w, h) {
        Ext.ux.senior.FileUploadField.superclass.onResize.call(this, w, h);
        this.wrap.setWidth(w);
    },

    // private
    onDestroy : function() {
        Ext.ux.senior.FileUploadField.superclass.onDestroy.call(this);
        Ext.destroy(this.fileInput, this.button, this.wrap);
    },

    onDisable : function() {
        Ext.ux.senior.FileUploadField.superclass.onDisable.call(this);
        this.doDisable(true);
        this.fileInput.addClass('x-item-disabled');
    },

    onEnable : function() {
        Ext.ux.senior.FileUploadField.superclass.onEnable.call(this);
        this.doDisable(false);
        this.fileInput.removeClass('x-item-disabled');

    },

    // private
    doDisable : function(disabled) {
        this.fileInput.dom.disabled = disabled;
    },

    // private
    preFocus : Ext.emptyFn,

    // private
    alignErrorIcon : function() {
        this.errorIcon.alignTo(this.wrap, 'tl-tr', [ 2, 0 ]);
    }

});

Ext.reg('fileuploadfield', Ext.ux.senior.FileUploadField);