Ext.namespace('Ext.ux.senior');

/**
 * Representa uma janela flutuante, que pode ser configurada para permitir ser redimensionada, mostrar botes para 
 * fechar  e maximizar, permitir mover sua posio.
 * 
 * Tambm pode ser configurada com uma Modal do framework, onde bloqueia o acesso dos componentes ao fundo e tem 
 * as funcionalidades de um fragmento Modal.
 * 
 * @author Patrick Nascimento
 */
Ext.ux.senior.Window = Ext.extend(Ext.Window, {

    /** Mostra ou esconde o boto de fechar. Padro  esconder o boto.*/
    closable : false,

    /** Ttulo da janela. */
    title : "&nbsp;",

    shadow : false,

    /** Limita a movimentao e o posicionamento da modal s bordas do container principal */ 
    constrain : true,
    
    /**
     * @override
     */
    initComponent : function() {

        if (this.owner && Ext.getCmp(this.owner).xtype == 'seniorviewport') {
            this.manager = Ext.ux.senior.TopWindowMgr;
        } else {
            this.manager = Ext.ux.senior.WindowMgr;
        }

        this.on('beforeclose', this.onBeforeClose);

        if (this.owner) {
            Ext.getCmp(this.owner).on('destroy', this.onOwnerDestroyed, this);
        }

        Ext.ux.senior.Window.superclass.initComponent.call(this, arguments);
    },

    onRender : function(ct, position) {
        Ext.ux.senior.Window.superclass.onRender.call(this, ct, position);

        if (this.owner) {
            if (this.modal) {

                var ownerCmp = Ext.getCmp(this.owner);
                if (ownerCmp.rendered) {
                    this.appendMask();
                } else {
                    ownerCmp.on('afterrender', this.appendMask, this);
                }
            }
        }
    },

    appendMask : function() {
        var ownerEl = Ext.getCmp(this.owner).el;
        this.mask.appendTo(ownerEl);
    },

    onOwnerDestroyed : function(owner) {
        this.close();
    },

    /**
     * @plugin EventManagerPlugin
     */
    eventMapping : function() {
        return {
            beforeclose : this.onClose,
            bodyresize : this.doLayout,
            resize : this.windowResize,
            move : this.onWindowMove,
            deactivate : this.windowClose,
            maximize : this.windowMaximize,
            restore : this.windowRestore
        };
    },

    /**
     * Evento de redimensionamento de uma janela flutuante
     * @param window janela flutuante redimensionada
     * @param w largura da janela
     * @param h altura da janela
     * @returns objeto de request de resize da janela
     */
    windowResize : function(window, w, h) {
        if (!this.maximized && this.modal && h != undefined && w != undefined) {

            return {
                type : 'windowResize',
                width : w,
                height : h
            };
        }
    },

    /**
     * Evento de movimentao de uma janela flutuante
     * @param window janela flutuante movida
     * @param x posio horizontal da janela
     * @param y posio vertical da janela
     * @returns objeto de request de movimentao da janela
     */
    onWindowMove : function(window, x, y) {
        if (!this.maximized && this.modal && x != undefined && y != undefined) {
            return {
                type : 'windowMove',
                posX : x,
                posY : y
            };
        }
    },

    /**
     * Manda fechar a window ao desativar ela, caso no seja uma modal.
     */
    windowClose : function() {
        if (!this.modal) {
            return {
                type : "closeWindow"
            };
        }
    },

    windowMaximize : function() {
        return {
            type : "windowMaximize"
        };
    },

    windowRestore : function() {

        var newHeight = this.height;
        var newWidth = this.width;

        if (this.height > window.document.height) {
            newHeight = window.document.height;
        }

        if (this.width > window.document.width) {
            newWidth = window.document.width;
        }

        // se o tamanho da window for maior que o tamanho do browser, deve ajustar para o tamanho do browser 

        if (newHeight != this.height || newWidth != this.width) {
            this.setSize(newWidth, newHeight);
        }

        // se a window ficar cortada ou escondida, deve ser centralizada.

        if (this.y < 0 || this.x < 0 || this.y + this.height > window.document.height || this.x + this.width > window.document.width) {
            this.center();
        }

        return {
            type : "windowRestore"
        };
    },

    /**
     * Informa o servidor para fechar essa Window.
     */
    onClose : function() {
        var eventName = this.modal === true ? "closeModal" : "closeWindow";

        return {
            type : eventName
        };
    },

    /**
     * Retorna false para cancelar o fechamento. 
     * 
     * O evento onClose se encarrega de enviar o close ao servidor, que por sua vez vai fechar a Window.
     */
    onBeforeClose : function() {
        return !this.modal;
    },

    /**
     * Posiciona a window nas coordenadas informadas.
     * Se a window ficar fora da tela, ela ser centralizada.
     * @param x a posio horizontal da window
     * @param y a posio vertical da window
     */
    setWindowPosition : function(x, y) {
        if (!this.maximized) {

            if (y < 0 || x < 0 || y + this.height > window.document.height || x + this.width > window.document.width) {
                this.center();
            } else {
                this.setPosition(x, y);
            }
        } else {
            // se a window estiver maximizada, a posio dela deve ser (0,0) por isso armazena a posio para que quando a window for restaurada, ela fique na posio setada.
            this.restorePos = [ x, y ];
        }
    }

});

Ext.reg('seniorwindow', Ext.ux.senior.Window);
