Ext.namespace('Ext.ux.senior');

/**
 * Desenha e controla um componente que tem a finalidade de repetir o contedo n vezes internamente conforme configurado
 * em <code>mimCount</code>.
 * 
 * @class Ext.ux.senior.Repeater
 * @extends Ext.ux.senior.Panel
 * @author Jean.Kirchner
 */
Ext.ux.senior.Repeater = Ext.extend(Ext.ux.senior.Panel, {

    hideBorders : true,
    border : false,

    /* Armazena elementos do scroll */
    pendentContent : false,
    emptyText : "",
    emptyElement : null,

    /* Altura mdia da linha no repetidor. */
    averageLineHeight : 0,

    lastScrollPos : 0,
    scrollEvent : null,
    scrollSent : false,
    scrollOrder : -1,

    isShowEmptyMessage : false,
    
    /**
     * Estas variveis so para controle de travamento do Repeater em uma situao onde muitos dados sem filtro algum so utilizados sem que os elementos renderizados ocupem espao, 
     * ou seja, esto invisveis.  uma forma mais simples de resolver o problema da tarefa <b>367301</b>.
     * O correto seria resolver no Servidor, impedindo que ocorra um problema de 'starvation' de mensagens menos prioritrias por vir mensagens mais prioritrias (mas isto no caso desta 
     * tarefa).
     */
    lastCount : 0,
    emptyUpdates : 0,

    /* Definio dos tipos de evento que o repeater pode enviar */
    eventType : {
        CONTENT : "CONTENT",
        NEXT : "NEXT",
        PREVIOUS : "PREVIOUS",
        PAGEUP : "PAGEUP",
        PAGEDOWN : "PAGEDOWN",
        LAST : "LAST",
        FIRST : "FIRST"
    },

    /**
     * Mapeamento dos eventos
     * 
     * @plugin EventManagerPlugin
     */
    eventMapping : function() {
        return {
            CONTENT : this.prepareContentEventData
        };
    },

    /**
     * Evento para pedir mais contedo
     * 
     * @event CONTENT
     */
    prepareContentEventData : function() {
        return {
            type : 'content'
        };
    },

    /**
     * Indica que tem contedo vindo, e at que acabe no faz mais requisies
     * 
     * @public
     */
    setPendentContent : function(a) {
        this.pendentContent = a;
    },

    /**
     * @public
     */
    setHasNext : function(a) {
        this.hasNext = a;
    },

    /**
     * @public
     */
    setHasPrior : function(a) {
        this.hasPrior = a;
    },

    /**
     * Adiciona listeners para os eventos de 'render' e 'afterrender'
     * 
     * @override
     */
    initComponent : function() {
        Ext.ux.senior.Repeater.superclass.initComponent.call(this);

        this.on('render', this.onRepeaterRender);
        this.on('afterrender', this.onAfterRepeaterRender);
        this.on('afterLayout', this.onAfterRepeaterLayout);
    },

    onAfterRepeaterLayout : function(ct, layout) {
        if (!this.hasListener('bodyresize')) {
            this.on('bodyresize', this.ajustScrollForceLayout, this);
        }
    },

    /**
     * Ao renderizar o repeater, cria e renderiza o elemento da mensagem que deve aparecer quando o repeater estiver vazio.
     * 
     * @private
     */
    onRepeaterRender : function(ct, position) {
        if (this.emptyText) {
            this.emptyElement = new Ext.ux.senior.form.Label({
                text : this.emptyText,
                cls : 'x-repeater-empty-message'
            });

            this.emptyElement.render(this.getEl());
        }
    },

    /**
     * @private
     */
    onAfterRepeaterRender : function() {
        this.body.addClass("x-repeater-body");

        /* Configura como 100% para fins de rolagem, no pode extender o tamanho */
        this.body.setHeight('100%');
        this.el.first().setHeight('100%');
        this.el.setHeight('100%');

        /* Devido a problemas de renderizao em algumas combinaes de layouts , no Safari, deve ser explicitado a largura  */
        if ((Ext.isSafari || Ext.isChrome) && this.initialConfig.width == null) {
            this.body.setWidth('100%');
            this.el.first().setWidth('100%');
            this.el.setWidth('100%');
        }

        /* Registro o evento de scroll para o elemento. */
        this.body.on('scroll', this.onScroll, this);

        /* Trata a mensagem de vazio do repeater */
        this.showEmptyMessage(this.isShowEmptyMessage);
    },

    ajustScrollForceLayout : function() {
        this.doLayout();
        this.adjustScroll();
    },

    /**
     * Retorna a altura do espao ocupado.
     * 
     * @private
     */
    getContentHeight : function() {
        var height = 0;
        this.items.each(function(item, index) {
            height += item.getHeight();
        }, this);
        return height;
    },

    /**
     * @public
     */
    adjustScroll : function() {
        if (!this.rendered) {
            return;
        }
        var itemCount = this.items ? this.items.getCount() : 0;
        var height = this.getInnerHeight();
        var contentHeight = this.getContentHeight();
        
        // Cada vez que aumentaram os itens sem ter contedo visvel, conta como um update "vazio". Aps 3 updates vazios, pra de pedir contedo.
        if (itemCount > this.lastCount) {
            if (contentHeight == 0) {
                this.emptyUpdates++;
            } else {
                this.emptyUpdates = 0;
            }
        }

        if ((itemCount == 0 || (contentHeight < height)) && (this.hasNext || this.hasPrior) && (this.emptyUpdates < 3)) {
            this.requestContent();
        } else {
            this.adjustScrollPos();
        }

        /*Seta o tamanho da tag PAI quando ocorre o resize para que acompanhe o tamanho*/
        var body = Ext.get(this.body.dom);
        body.setHeight("100%");

        this.onScroll();
        
        this.lastCount = itemCount;
    },

    setShowEmptyMessage : function(show) {
        this.isShowEmptyMessage = show;
        this.showEmptyMessage(this.isShowEmptyMessage);
    },

    /**
     * @private
     */
    showEmptyMessage : function(show) {
        if (this.emptyElement) {
            this.emptyElement.setVisible(show);
        }
    },

    /**
     * Recebe o evento de scroll.
     * 
     * @event scroll
     */
    onScroll : function() {
        this.scrollSent = false;
        var scroll = this.body.dom.scrollTop;
        var forward = (scroll - this.lastScrollPos) >= 0;
        if (!forward) {
            this.scrollEvent = scroll == 0 ? this.eventType.FIRST : this.eventType.PAGEUP;
        } else {
            this.scrollEvent = scroll == this.getInnerHeight() ? this.eventType.LAST : this.eventType.PAGEDOWN;
            if (this.hasNext) {
                var contentHeight = this.getContentHeight();
                var bodyHeight = this.getInnerHeight();
                /*  80% do contedo seria o ponto de requisitar contedo */
                var fetchPoint = (contentHeight * 0.80);
                /* se tiver mais contedo e o scroll + o tamanho da rea visvel chegar no ponto de fetch, enviar request */
                if (scroll + bodyHeight >= fetchPoint) {
                    this.requestContent();
                }
            }
        }        
    },

    /**
     * @private
     */
    adjustScrollPos : function() {
        var dif = this.getInnerHeight() - this.getContentHeight();
        this.lastScrollPos = this.body.dom.scrollTop;
        if (dif == this.lastScrollPos && this.hasNext) {
            this.body.dom.scrollTop -= dif * 0.25;
        } else if (this.lastScrollPos == 0 && this.hasPrior) {
            this.body.dom.scrollTop += dif * 0.25;
        }
        this.lastScrollPos = this.body.dom.scrollTop;
    },

    /**
     * Envia uma requisio para o servidor pedindo contedo.
     * 
     * @private
     */
    requestContent : function() {
        if (this.pendentContent == false) {
            this.pendentContent = true;
            this.fireEvent(this.eventType.CONTENT);
        }
    },

    /**
     * @override
     */
    destroy : function(e) {
        Ext.ux.senior.Repeater.superclass.destroy.call(this);
        Ext.destroy(this.emptyElement);
        delete this.emptyElement;
    }

});

Ext.reg('seniorrepeater', Ext.ux.senior.Repeater);